/*
 * Decompiled with CFR 0.152.
 */
package org.springframework.boot.logging;

import java.io.IOException;
import java.util.Arrays;
import java.util.Collections;
import java.util.EnumSet;
import java.util.HashSet;
import java.util.IdentityHashMap;
import java.util.Objects;
import java.util.Set;
import java.util.function.BiPredicate;
import java.util.function.Function;
import java.util.function.Predicate;
import java.util.function.ToIntFunction;
import org.springframework.boot.logging.StackTracePrinter;
import org.springframework.util.Assert;

public final class StandardStackTracePrinter
implements StackTracePrinter {
    private static final String DEFAULT_LINE_SEPARATOR = System.lineSeparator();
    private static final ToIntFunction<StackTraceElement> DEFAULT_FRAME_HASHER = frame -> Objects.hash(frame.getClassName(), frame.getMethodName(), frame.getLineNumber());
    private static final int UNLIMITED = Integer.MAX_VALUE;
    private final EnumSet<Option> options;
    private final int maximumLength;
    private final String lineSeparator;
    private final Predicate<Throwable> filter;
    private final BiPredicate<Integer, StackTraceElement> frameFilter;
    private final Function<Throwable, String> formatter;
    private final Function<StackTraceElement, String> frameFormatter;
    private final ToIntFunction<StackTraceElement> frameHasher;

    private StandardStackTracePrinter(EnumSet<Option> options, int maximumLength, String lineSeparator, Predicate<Throwable> filter2, BiPredicate<Integer, StackTraceElement> frameFilter, Function<Throwable, String> formatter, Function<StackTraceElement, String> frameFormatter, ToIntFunction<StackTraceElement> frameHasher) {
        this.options = options;
        this.maximumLength = maximumLength;
        this.lineSeparator = lineSeparator != null ? lineSeparator : DEFAULT_LINE_SEPARATOR;
        this.filter = filter2 != null ? filter2 : t -> true;
        this.frameFilter = frameFilter != null ? frameFilter : (i2, t) -> true;
        this.formatter = formatter != null ? formatter : Object::toString;
        this.frameFormatter = frameFormatter != null ? frameFormatter : Object::toString;
        this.frameHasher = frameHasher;
    }

    @Override
    public void printStackTrace(Throwable throwable, Appendable out) throws IOException {
        if (this.filter.test(throwable)) {
            Set<Throwable> seen = Collections.newSetFromMap(new IdentityHashMap());
            Output output = new Output(out);
            Print print = new Print("", "", output);
            this.printFullStackTrace(seen, print, new StackTrace(throwable), null);
        }
    }

    private void printFullStackTrace(Set<Throwable> seen, Print print, StackTrace stackTrace, StackTrace enclosing) throws IOException {
        if (stackTrace == null) {
            return;
        }
        if (!seen.add(stackTrace.throwable())) {
            String hashPrefix = stackTrace.hashPrefix(this.frameHasher);
            String throwable = this.formatter.apply(stackTrace.throwable());
            print.circularReference(hashPrefix, throwable);
            return;
        }
        StackTrace cause = stackTrace.cause();
        if (!this.hasOption(Option.ROOT_FIRST)) {
            this.printSingleStackTrace(seen, print, stackTrace, enclosing);
            this.printFullStackTrace(seen, print.withCausedByCaption(cause), cause, stackTrace);
        } else {
            this.printFullStackTrace(seen, print, cause, stackTrace);
            this.printSingleStackTrace(seen, print.withWrappedByCaption(cause), stackTrace, enclosing);
        }
    }

    private void printSingleStackTrace(Set<Throwable> seen, Print print, StackTrace stackTrace, StackTrace enclosing) throws IOException {
        String hashPrefix = stackTrace.hashPrefix(this.frameHasher);
        String throwable = this.formatter.apply(stackTrace.throwable());
        print.thrown(hashPrefix, throwable);
        this.printFrames(print, stackTrace, enclosing);
        if (!this.hasOption(Option.HIDE_SUPPRESSED)) {
            for (StackTrace suppressed : stackTrace.suppressed()) {
                this.printFullStackTrace(seen, print.withSuppressedCaption(), suppressed, stackTrace);
            }
        }
    }

    private void printFrames(Print print, StackTrace stackTrace, StackTrace enclosing) throws IOException {
        int commonFrames = !this.hasOption(Option.SHOW_COMMON_FRAMES) ? stackTrace.commonFramesCount(enclosing) : 0;
        int filteredFrames = 0;
        for (int i2 = 0; i2 < stackTrace.frames().length - commonFrames; ++i2) {
            StackTraceElement element = stackTrace.frames()[i2];
            if (!this.frameFilter.test(i2, element)) {
                ++filteredFrames;
                continue;
            }
            print.omittedFilteredFrames(filteredFrames);
            filteredFrames = 0;
            print.at(this.frameFormatter.apply(element));
        }
        print.omittedFilteredFrames(filteredFrames);
        if (commonFrames != 0) {
            print.omittedCommonFrames(commonFrames);
        }
    }

    public StandardStackTracePrinter withCommonFrames() {
        return this.withOption(Option.SHOW_COMMON_FRAMES);
    }

    public StandardStackTracePrinter withoutSuppressed() {
        return this.withOption(Option.HIDE_SUPPRESSED);
    }

    public StandardStackTracePrinter withMaximumLength(int maximumLength) {
        Assert.isTrue(maximumLength > 0, "'maximumLength' must be positive");
        return new StandardStackTracePrinter(this.options, maximumLength, this.lineSeparator, this.filter, this.frameFilter, this.formatter, this.frameFormatter, this.frameHasher);
    }

    public StandardStackTracePrinter withMaximumThrowableDepth(int maximumThrowableDepth) {
        Assert.isTrue(maximumThrowableDepth > 0, "'maximumThrowableDepth' must be positive");
        return this.withFrameFilter((index, element) -> index < maximumThrowableDepth);
    }

    public StandardStackTracePrinter withFilter(Predicate<Throwable> predicate) {
        Assert.notNull(predicate, "'predicate' must not be null");
        return new StandardStackTracePrinter(this.options, this.maximumLength, this.lineSeparator, this.filter.and(predicate), this.frameFilter, this.formatter, this.frameFormatter, this.frameHasher);
    }

    public StandardStackTracePrinter withFrameFilter(BiPredicate<Integer, StackTraceElement> predicate) {
        Assert.notNull(predicate, "'predicate' must not be null");
        return new StandardStackTracePrinter(this.options, this.maximumLength, this.lineSeparator, this.filter, this.frameFilter.and(predicate), this.formatter, this.frameFormatter, this.frameHasher);
    }

    public StandardStackTracePrinter withLineSeparator(String lineSeparator) {
        Assert.notNull((Object)lineSeparator, "'lineSeparator' must not be null");
        return new StandardStackTracePrinter(this.options, this.maximumLength, lineSeparator, this.filter, this.frameFilter, this.formatter, this.frameFormatter, this.frameHasher);
    }

    public StandardStackTracePrinter withFormatter(Function<Throwable, String> formatter) {
        Assert.notNull(formatter, "'formatter' must not be null");
        return new StandardStackTracePrinter(this.options, this.maximumLength, this.lineSeparator, this.filter, this.frameFilter, formatter, this.frameFormatter, this.frameHasher);
    }

    public StandardStackTracePrinter withFrameFormatter(Function<StackTraceElement, String> frameFormatter) {
        Assert.notNull(frameFormatter, "'frameFormatter' must not be null");
        return new StandardStackTracePrinter(this.options, this.maximumLength, this.lineSeparator, this.filter, this.frameFilter, this.formatter, frameFormatter, this.frameHasher);
    }

    public StandardStackTracePrinter withHashes() {
        return this.withHashes(true);
    }

    public StandardStackTracePrinter withHashes(boolean hashes) {
        return this.withHashes(!hashes ? null : DEFAULT_FRAME_HASHER);
    }

    public StandardStackTracePrinter withHashes(ToIntFunction<StackTraceElement> frameHasher) {
        return new StandardStackTracePrinter(this.options, this.maximumLength, this.lineSeparator, this.filter, this.frameFilter, this.formatter, this.frameFormatter, frameHasher);
    }

    private StandardStackTracePrinter withOption(Option option) {
        EnumSet<Option> options = EnumSet.copyOf(this.options);
        options.add(option);
        return new StandardStackTracePrinter(options, this.maximumLength, this.lineSeparator, this.filter, this.frameFilter, this.formatter, this.frameFormatter, this.frameHasher);
    }

    private boolean hasOption(Option option) {
        return this.options.contains((Object)option);
    }

    public static StandardStackTracePrinter rootLast() {
        return new StandardStackTracePrinter(EnumSet.noneOf(Option.class), Integer.MAX_VALUE, null, null, null, null, null, null);
    }

    public static StandardStackTracePrinter rootFirst() {
        return new StandardStackTracePrinter(EnumSet.of(Option.ROOT_FIRST), Integer.MAX_VALUE, null, null, null, null, null, null);
    }

    private class Output {
        private static final String ELLIPSIS = "...";
        private final Appendable out;
        private int remaining;

        Output(Appendable out) {
            this.out = out;
            this.remaining = StandardStackTracePrinter.this.maximumLength - ELLIPSIS.length();
        }

        void println(String indent, String string) throws IOException {
            if (this.remaining > 0) {
                String line = indent + string + StandardStackTracePrinter.this.lineSeparator;
                if (line.length() > this.remaining) {
                    line = line.substring(0, this.remaining) + ELLIPSIS;
                }
                this.out.append(line);
                this.remaining -= line.length();
            }
        }
    }

    private record Print(String indent, String caption, Output output) {
        void circularReference(String hashPrefix, String throwable) throws IOException {
            this.output.println(this.indent, this.caption + "[CIRCULAR REFERENCE: " + hashPrefix + throwable + "]");
        }

        void thrown(String hashPrefix, String throwable) throws IOException {
            this.output.println(this.indent, this.caption + hashPrefix + throwable);
        }

        void at(String frame) throws IOException {
            this.output.println(this.indent, "\tat " + frame);
        }

        void omittedFilteredFrames(int filteredFrameCount) throws IOException {
            if (filteredFrameCount > 0) {
                this.output.println(this.indent, "\t... " + filteredFrameCount + " filtered");
            }
        }

        void omittedCommonFrames(int commonFrameCount) throws IOException {
            this.output.println(this.indent, "\t... " + commonFrameCount + " more");
        }

        Print withCausedByCaption(StackTrace causedBy) {
            return this.withCaption(causedBy != null, "", "Caused by: ");
        }

        Print withWrappedByCaption(StackTrace wrappedBy) {
            return this.withCaption(wrappedBy != null, "", "Wrapped by: ");
        }

        public Print withSuppressedCaption() {
            return this.withCaption(true, "\t", "Suppressed: ");
        }

        private Print withCaption(boolean test, String extraIndent, String caption) {
            return test ? new Print(this.indent + extraIndent, caption, this.output) : this;
        }
    }

    private static final class StackTrace {
        private final Throwable throwable;
        private final StackTraceElement[] frames;
        private StackTrace[] suppressed;
        private StackTrace cause;
        private Integer hash;
        private String hashPrefix;

        private StackTrace(Throwable throwable) {
            this.throwable = throwable;
            this.frames = throwable != null ? throwable.getStackTrace() : null;
        }

        Throwable throwable() {
            return this.throwable;
        }

        StackTraceElement[] frames() {
            return this.frames;
        }

        int commonFramesCount(StackTrace other) {
            if (other == null) {
                return 0;
            }
            int index = this.frames.length - 1;
            for (int otherIndex = other.frames.length - 1; index >= 0 && otherIndex >= 0 && this.frames[index].equals(other.frames[otherIndex]); --index, --otherIndex) {
            }
            return this.frames.length - 1 - index;
        }

        StackTrace[] suppressed() {
            if (this.suppressed == null && this.throwable != null) {
                this.suppressed = (StackTrace[])Arrays.stream(this.throwable.getSuppressed()).map(StackTrace::new).toArray(StackTrace[]::new);
            }
            return this.suppressed;
        }

        StackTrace cause() {
            if (this.cause == null && this.throwable != null) {
                Throwable cause = this.throwable.getCause();
                this.cause = cause != null ? new StackTrace(cause) : null;
            }
            return this.cause;
        }

        String hashPrefix(ToIntFunction<StackTraceElement> frameHasher) {
            if (frameHasher == null || this.throwable() == null) {
                return "";
            }
            this.hashPrefix = this.hashPrefix != null ? this.hashPrefix : String.format("<#%08x> ", this.hash(new HashSet<Throwable>(), frameHasher));
            return this.hashPrefix;
        }

        private int hash(HashSet<Throwable> seen, ToIntFunction<StackTraceElement> frameHasher) {
            if (this.hash != null) {
                return this.hash;
            }
            int hash = 0;
            if (this.cause() != null && seen.add(this.cause().throwable())) {
                hash = this.cause().hash(seen, frameHasher);
            }
            hash = 31 * hash + this.throwable().getClass().getName().hashCode();
            for (StackTraceElement frame : this.frames()) {
                hash = 31 * hash + frameHasher.applyAsInt(frame);
            }
            this.hash = hash;
            return hash;
        }
    }

    private static enum Option {
        ROOT_FIRST,
        SHOW_COMMON_FRAMES,
        HIDE_SUPPRESSED;

    }
}

