/*
 * Decompiled with CFR 0.152.
 */
package com.hazelcast.internal.rest.security;

import com.hazelcast.config.rest.RestConfig;
import com.hazelcast.internal.util.Preconditions;
import com.hazelcast.logging.ILogger;
import com.hazelcast.logging.Logger;
import jakarta.annotation.PreDestroy;
import java.time.Clock;
import java.time.Duration;
import java.time.LocalDateTime;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.logging.Level;
import javax.security.auth.login.LoginException;
import org.springframework.context.annotation.Lazy;
import org.springframework.stereotype.Service;

@Service
public class LockoutService {
    private static final ILogger LOGGER = Logger.getLogger(LockoutService.class);
    private static final Duration CLEANUP_DELAY = Duration.ofSeconds(10L);
    private final RestConfig restConfig;
    private final ScheduledExecutorService executor;
    private volatile boolean initialized;
    private int maxAttempts;
    private Duration lockoutDuration;
    private boolean unlimitedMaxAttempts;
    private final Map<String, LoginAttemptInfo> attemptCache = new ConcurrentHashMap<String, LoginAttemptInfo>();
    private Clock clock = Clock.systemDefaultZone();

    public LockoutService(@Lazy RestConfig restConfig) {
        this.restConfig = restConfig;
        this.executor = Executors.newSingleThreadScheduledExecutor();
        this.executor.scheduleWithFixedDelay(() -> {
            this.ensureInitialized();
            LocalDateTime now = LocalDateTime.now(this.clock);
            this.attemptCache.entrySet().removeIf(e -> ((LoginAttemptInfo)e.getValue()).expiresAt.isBefore(now));
        }, CLEANUP_DELAY.getSeconds() * 2L, CLEANUP_DELAY.getSeconds(), TimeUnit.SECONDS);
        Holder.lockoutService = this;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void ensureInitialized() {
        if (!this.initialized) {
            LockoutService lockoutService = this;
            synchronized (lockoutService) {
                if (!this.initialized) {
                    this.maxAttempts = this.restConfig.getMaxLoginAttempts();
                    Preconditions.checkNotNegative((int)this.maxAttempts, (String)"Max Login Attempts cannot be negative");
                    this.lockoutDuration = this.restConfig.getLockoutDuration();
                    this.unlimitedMaxAttempts = this.maxAttempts == 0;
                    this.initialized = true;
                }
            }
        }
    }

    @PreDestroy
    public void shutdown() {
        this.executor.shutdown();
        Holder.lockoutService = null;
    }

    public void ensureNotLockedOut(String username) throws LoginException {
        this.ensureInitialized();
        if (this.unlimitedMaxAttempts) {
            return;
        }
        LoginAttemptInfo lockoutInfo = this.attemptCache.get(username);
        if (lockoutInfo == null) {
            return;
        }
        if (lockoutInfo.expiresAt.isBefore(LocalDateTime.now(this.clock))) {
            this.attemptCache.remove(username);
            return;
        }
        if (lockoutInfo.attempts >= this.maxAttempts) {
            throw new LoginException("Account is locked out due to too many unsuccessful login attempts.");
        }
    }

    public void markFailedLogin(String username) {
        this.ensureInitialized();
        if (this.unlimitedMaxAttempts) {
            return;
        }
        this.attemptCache.compute(username, (t, old) -> {
            int attemptNo = old == null ? 1 : old.attempts + 1;
            LOGGER.log(Level.WARNING, "Login attempt #" + attemptNo + " failed for user '" + username + "'.");
            return new LoginAttemptInfo(username, LocalDateTime.now(this.clock).plus(this.lockoutDuration), attemptNo);
        });
    }

    public void markSuccessfulLogin(String username) throws LoginException {
        this.ensureInitialized();
        if (this.unlimitedMaxAttempts) {
            return;
        }
        this.ensureNotLockedOut(username);
        this.attemptCache.remove(username);
        LOGGER.log(Level.FINE, "Successful login of user '" + username + "'");
    }

    static void setClock(Clock clock) {
        Holder.lockoutService.clock = clock;
    }

    static class Holder {
        static LockoutService lockoutService;

        Holder() {
        }
    }

    private record LoginAttemptInfo(String username, LocalDateTime expiresAt, int attempts) {
    }
}

