/*
 * Decompiled with CFR 0.152.
 */
package com.hazelcast.cp.internal.raft.impl;

import com.hazelcast.core.HazelcastException;
import com.hazelcast.cp.internal.raft.impl.RaftEndpoint;
import com.hazelcast.cp.internal.raft.impl.RaftNodeImpl;
import com.hazelcast.cp.internal.raft.impl.dto.AppendSuccessResponse;
import com.hazelcast.cp.internal.raft.impl.dto.InstallSnapshotRequest;
import com.hazelcast.cp.internal.raft.impl.dto.InstallSnapshotResponse;
import com.hazelcast.cp.internal.raft.impl.log.SnapshotChunk;
import com.hazelcast.cp.internal.raft.impl.log.SnapshotEntry;
import com.hazelcast.cp.internal.raft.impl.persistence.RaftStateStore;
import com.hazelcast.cp.internal.raft.impl.state.RaftState;
import com.hazelcast.internal.util.CollectionUtil;
import com.hazelcast.internal.util.IterableUtil;
import com.hazelcast.logging.ILogger;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Comparator;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Set;
import java.util.stream.IntStream;

public class ChunkedSnapshotInstaller {
    private static final int UNSET = -1;
    private final RaftNodeImpl raftNode;
    private final RaftStateStore stateStore;
    private final ILogger logger;
    private long currentSnapshotIndex;
    private int currentTerm;
    private int totalChunks;
    private List<SnapshotChunk> receivedChunks;
    private Set<Integer> missingChunks;
    private Collection<RaftEndpoint> groupMembers;
    private long groupMembersLogIndex;

    public ChunkedSnapshotInstaller(RaftNodeImpl raftNode) {
        this.raftNode = raftNode;
        this.logger = raftNode.getLogger(this.getClass());
        this.stateStore = raftNode.state().stateStore();
    }

    public void processChunk(InstallSnapshotRequest request) {
        if (ChunkProcessingDecision.ABORT == this.validateAndStoreChunk(request)) {
            return;
        }
        if (this.hasMissingChunks()) {
            this.requestMissingChunks(request);
            return;
        }
        try {
            this.installCompleteSnapshot();
            this.notifySuccess(request);
            this.logger.info(String.format("Successfully installed snapshot [index=%d, term=%d] from leader %s", this.currentSnapshotIndex, this.currentTerm, request.leader()));
        }
        catch (Throwable t) {
            this.logger.severe(String.format("Failed to install snapshot [index=%d, term=%d]: %s", this.currentSnapshotIndex, this.currentTerm, t.getMessage()));
            throw t;
        }
        finally {
            this.reset();
        }
    }

    private ChunkProcessingDecision validateAndStoreChunk(InstallSnapshotRequest request) {
        SnapshotChunk chunk = request.snapshotChunk();
        if (this.isStaleChunk(chunk)) {
            this.logger.fine("Ignoring stale chunk - received index %d is behind our current commit index %d", chunk.index(), this.raftNode.state().commitIndex());
            return ChunkProcessingDecision.ABORT;
        }
        if (this.isDuplicateCommitIndex(chunk)) {
            this.notifySuccess(request);
            this.logger.fine("Skipping chunk - index %d matches our current commit index %d (already applied)", chunk.index(), this.raftNode.state().commitIndex());
            return ChunkProcessingDecision.ABORT;
        }
        if (this.isFromOlderSnapshot(chunk)) {
            this.logger.fine("Rejecting old snapshot chunk - received snapshot [term=%d, index=%d, chunk=%d(0-based)/%d], is older than current snapshot index: %d", chunk.term(), chunk.index(), chunk.chunkNumber(), chunk.chunkCount(), this.currentSnapshotIndex);
            return ChunkProcessingDecision.ABORT;
        }
        if (this.isFromNewerSnapshot(chunk)) {
            this.logger.info(String.format("Starting new snapshot installation - switching from index %d to newer index %d", this.currentSnapshotIndex, chunk.index()));
            this.handleNewSnapshotSeries(chunk);
        }
        if (chunk.operation() == null) {
            this.logger.fine("Skip metadata chunk [index: %d, chunk: %d (0-based)] (triggers snapshot fetch, may resend on retry). ", chunk.index(), chunk.chunkNumber());
            return ChunkProcessingDecision.CONTINUE;
        }
        if (!this.isTermConsistent(chunk)) {
            throw new IllegalStateException(String.format("Term mismatch detected! Snapshot index %d expects term %d but received term %d. This indicates a serious consistency issue.", chunk.index(), this.currentTerm, chunk.term()));
        }
        if (!this.missingChunks.remove(chunk.chunkNumber())) {
            this.logger.fine("Received duplicate chunk %d (0-based) for snapshot index %d term %d (current progress: %d of %d chunks received, %d still missing)", chunk.chunkNumber(), chunk.index(), chunk.term(), this.receivedChunks.size(), this.totalChunks, this.missingChunks.size());
            return ChunkProcessingDecision.CONTINUE;
        }
        this.persistAndTrackChunk(chunk);
        this.logger.info(String.format("Successfully stored chunk %d (0-based) of %d for snapshot index %d term %d (%d of %d chunks complete, %d remaining)", chunk.chunkNumber(), this.totalChunks, this.currentSnapshotIndex, this.currentTerm, this.receivedChunks.size(), this.totalChunks, this.missingChunks.size()));
        return ChunkProcessingDecision.CONTINUE;
    }

    private void installCompleteSnapshot() {
        this.raftNode.getRaftIntegration().resetForChunkedSnapshotRestore();
        SnapshotEntry snapshot = this.createSnapshotFromChunks();
        this.raftNode.installSnapshot(snapshot);
    }

    private void requestMissingChunks(InstallSnapshotRequest request) {
        assert (this.hasMissingChunks()) : "always checked before method call";
        int nextMissingChunkNumber = IterableUtil.getFirst(this.missingChunks, -1);
        assert (nextMissingChunkNumber != -1) : "never expect an unset missingChunkNumber";
        RaftState state = this.raftNode.state();
        InstallSnapshotResponse response = new InstallSnapshotResponse(state.localEndpoint(), state.term(), request.queryRound(), request.flowControlSequenceNumber(), this.currentSnapshotIndex, nextMissingChunkNumber);
        this.raftNode.getRaftIntegration().send(response, this.raftNode.getLeader());
    }

    private boolean isStaleChunk(SnapshotChunk chunk) {
        return chunk.index() < this.raftNode.state().commitIndex();
    }

    private boolean isDuplicateCommitIndex(SnapshotChunk chunk) {
        return chunk.index() == this.raftNode.state().commitIndex();
    }

    private boolean isFromOlderSnapshot(SnapshotChunk chunk) {
        return chunk.index() < this.currentSnapshotIndex;
    }

    private boolean isFromNewerSnapshot(SnapshotChunk chunk) {
        return chunk.index() > this.currentSnapshotIndex;
    }

    private boolean isTermConsistent(SnapshotChunk chunk) {
        return chunk.term() == this.currentTerm;
    }

    private void handleNewSnapshotSeries(SnapshotChunk chunk) {
        this.cleanupPreviousSnapshot();
        this.initialize(chunk.index(), chunk.term(), chunk.chunkCount(), chunk.groupMembersLogIndex(), chunk.groupMembers());
    }

    private void persistAndTrackChunk(SnapshotChunk chunk) {
        assert (chunk.operation() != null) : "Can never be null " + String.valueOf(chunk);
        try {
            this.stateStore.persistSnapshotChunk(chunk);
            this.receivedChunks.add(chunk);
        }
        catch (IOException e) {
            this.missingChunks.add(chunk.chunkNumber());
            this.logger.severe(String.format("Failed to persist snapshot chunk [index=%d, term=%d, chunk=%d(0-based)]: %s", chunk.index(), chunk.term(), chunk.chunkNumber(), e.getMessage()), e);
            throw new HazelcastException(e);
        }
    }

    private void reset() {
        this.initialize(-1L, -1, -1, -1L, null);
    }

    private void initialize(long index, int term, int chunkCount, long membersLogIndex, Collection<RaftEndpoint> members) {
        this.currentSnapshotIndex = index;
        this.currentTerm = term;
        this.totalChunks = chunkCount;
        this.groupMembersLogIndex = membersLogIndex;
        this.groupMembers = members;
        this.receivedChunks = new ArrayList<SnapshotChunk>();
        this.missingChunks = new LinkedHashSet<Integer>();
        IntStream.range(0, chunkCount).forEach(this.missingChunks::add);
    }

    private void cleanupPreviousSnapshot() {
        if (!CollectionUtil.isEmpty(this.receivedChunks)) {
            try {
                this.stateStore.deleteSnapshotChunks(this.currentSnapshotIndex);
            }
            catch (IOException e) {
                this.logger.warning(String.format("Failed to delete previous snapshot chunks for index %d: %s", this.currentSnapshotIndex, e.getMessage()), e);
                throw new HazelcastException(e);
            }
        }
    }

    private SnapshotEntry createSnapshotFromChunks() {
        this.receivedChunks.sort(Comparator.comparingInt(SnapshotChunk::chunkNumber));
        try {
            this.stateStore.flushLogs();
        }
        catch (IOException e) {
            throw new HazelcastException(e);
        }
        return new SnapshotEntry(this.currentTerm, this.currentSnapshotIndex, this.receivedChunks, this.groupMembersLogIndex, this.groupMembers);
    }

    private void notifySuccess(InstallSnapshotRequest request) {
        AppendSuccessResponse response = new AppendSuccessResponse(this.raftNode.getLocalMember(), request.term(), request.snapshotChunk().index(), request.queryRound(), request.flowControlSequenceNumber());
        this.raftNode.send(response, request.leader());
    }

    private boolean hasMissingChunks() {
        return !this.missingChunks.isEmpty();
    }

    List<SnapshotChunk> getReceivedChunks() {
        return this.receivedChunks;
    }

    private static enum ChunkProcessingDecision {
        CONTINUE,
        ABORT;

    }
}

