/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.util.curator;

import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.LinkedList;
import java.util.List;
import org.apache.commons.lang3.StringUtils;
import org.apache.curator.RetryPolicy;
import org.apache.curator.framework.AuthInfo;
import org.apache.curator.framework.CuratorFramework;
import org.apache.curator.framework.CuratorFrameworkFactory;
import org.apache.curator.framework.api.ACLBackgroundPathAndBytesable;
import org.apache.curator.framework.api.ACLPathAndBytesable;
import org.apache.curator.framework.api.BackgroundPathAndBytesable;
import org.apache.curator.framework.api.PathAndBytesable;
import org.apache.curator.framework.api.WatchPathable;
import org.apache.curator.framework.api.transaction.CuratorOp;
import org.apache.curator.retry.RetryNTimes;
import org.apache.curator.utils.ZookeeperFactory;
import org.apache.hadoop.classification.InterfaceAudience;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.security.SecurityUtil;
import org.apache.hadoop.security.authentication.util.JaasConfiguration;
import org.apache.hadoop.util.Preconditions;
import org.apache.hadoop.util.ZKUtil;
import org.apache.zookeeper.CreateMode;
import org.apache.zookeeper.Watcher;
import org.apache.zookeeper.ZooKeeper;
import org.apache.zookeeper.client.ZKClientConfig;
import org.apache.zookeeper.data.ACL;
import org.apache.zookeeper.data.Stat;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@InterfaceAudience.Private
public final class ZKCuratorManager {
    private static final Logger LOG = LoggerFactory.getLogger(ZKCuratorManager.class);
    private final Configuration conf;
    private CuratorFramework curator;

    public ZKCuratorManager(Configuration config) throws IOException {
        this.conf = config;
    }

    public CuratorFramework getCurator() {
        return this.curator;
    }

    public void close() {
        if (this.curator != null) {
            this.curator.close();
        }
    }

    public static List<ACL> getZKAcls(Configuration conf) throws IOException {
        String zkAclConf = conf.get("hadoop.zk.acl", "world:anyone:rwcda");
        try {
            zkAclConf = ZKUtil.resolveConfIndirection(zkAclConf);
            return ZKUtil.parseACLs(zkAclConf);
        }
        catch (IOException | ZKUtil.BadAclFormatException e) {
            LOG.error("Couldn't read ACLs based on {}", (Object)"hadoop.zk.acl");
            throw e;
        }
    }

    public static List<ZKUtil.ZKAuthInfo> getZKAuths(Configuration conf) throws IOException {
        return SecurityUtil.getZKAuthInfos(conf, "hadoop.zk.auth");
    }

    public void start() throws IOException {
        this.start(new ArrayList<AuthInfo>());
    }

    public void start(List<AuthInfo> authInfos) throws IOException {
        this.start(authInfos, false);
    }

    public void start(List<AuthInfo> authInfos, boolean sslEnabled) throws IOException {
        ZKClientConfig zkClientConfig = new ZKClientConfig();
        String zkHostPort = this.conf.get("hadoop.zk.address");
        if (zkHostPort == null) {
            throw new IOException("hadoop.zk.address is not configured.");
        }
        LOG.debug("Configured {} as {}", (Object)"hadoop.zk.address", (Object)zkHostPort);
        int numRetries = this.conf.getInt("hadoop.zk.num-retries", 1000);
        int zkSessionTimeout = this.conf.getInt("hadoop.zk.timeout-ms", 10000);
        int zkRetryInterval = this.conf.getInt("hadoop.zk.retry-interval-ms", 1000);
        RetryNTimes retryPolicy = new RetryNTimes(numRetries, zkRetryInterval);
        List<ZKUtil.ZKAuthInfo> zkAuths = ZKCuratorManager.getZKAuths(this.conf);
        if (authInfos == null) {
            authInfos = new ArrayList<AuthInfo>();
        }
        for (ZKUtil.ZKAuthInfo zkAuth : zkAuths) {
            authInfos.add(new AuthInfo(zkAuth.getScheme(), zkAuth.getAuth()));
        }
        if (sslEnabled) {
            this.validateSslConfiguration(this.conf);
        }
        CuratorFramework client = CuratorFrameworkFactory.builder().connectString(zkHostPort).zookeeperFactory((ZookeeperFactory)new HadoopZookeeperFactory(this.conf.get("hadoop.zk.server.principal"), this.conf.get("hadoop.zk.kerberos.principal"), this.conf.get("hadoop.zk.kerberos.keytab"), sslEnabled, new SecurityUtil.TruststoreKeystore(this.conf))).zkClientConfig(zkClientConfig).sessionTimeoutMs(zkSessionTimeout).retryPolicy((RetryPolicy)retryPolicy).authorization(authInfos).build();
        client.start();
        this.curator = client;
    }

    private void validateSslConfiguration(Configuration config) throws IOException {
        if (StringUtils.isEmpty(config.get("hadoop.zk.ssl.keystore.location"))) {
            throw new IOException("The SSL encryption is enabled for the component's ZooKeeper client connection, however the hadoop.zk.ssl.keystore.location parameter is empty.");
        }
        if (StringUtils.isEmpty(config.get("hadoop.zk.ssl.keystore.password"))) {
            throw new IOException("The SSL encryption is enabled for the component's ZooKeeper client connection, however the hadoop.zk.ssl.keystore.password parameter is empty.");
        }
        if (StringUtils.isEmpty(config.get("hadoop.zk.ssl.truststore.location"))) {
            throw new IOException("The SSL encryption is enabled for the component's ZooKeeper client connection, however the hadoop.zk.ssl.truststore.location parameter is empty.");
        }
        if (StringUtils.isEmpty(config.get("hadoop.zk.ssl.truststore.password"))) {
            throw new IOException("The SSL encryption is enabled for the component's ZooKeeper client connection, however the hadoop.zk.ssl.truststore.password  parameter is empty.");
        }
    }

    public List<ACL> getACL(String path) throws Exception {
        return (List)this.curator.getACL().forPath(path);
    }

    public byte[] getData(String path) throws Exception {
        return (byte[])this.curator.getData().forPath(path);
    }

    public byte[] getData(String path, Stat stat) throws Exception {
        return (byte[])((WatchPathable)this.curator.getData().storingStatIn(stat)).forPath(path);
    }

    public String getStringData(String path) throws Exception {
        byte[] bytes = this.getData(path);
        if (bytes != null) {
            return new String(bytes, StandardCharsets.UTF_8);
        }
        return null;
    }

    public String getStringData(String path, Stat stat) throws Exception {
        byte[] bytes = this.getData(path, stat);
        if (bytes != null) {
            return new String(bytes, StandardCharsets.UTF_8);
        }
        return null;
    }

    public void setData(String path, byte[] data, int version) throws Exception {
        ((BackgroundPathAndBytesable)this.curator.setData().withVersion(version)).forPath(path, data);
    }

    public void setData(String path, String data, int version) throws Exception {
        byte[] bytes = data.getBytes(StandardCharsets.UTF_8);
        this.setData(path, bytes, version);
    }

    public List<String> getChildren(String path) throws Exception {
        return (List)this.curator.getChildren().forPath(path);
    }

    public boolean exists(String path) throws Exception {
        return this.curator.checkExists().forPath(path) != null;
    }

    public boolean create(String path) throws Exception {
        return this.create(path, null);
    }

    public boolean create(String path, List<ACL> zkAcl) throws Exception {
        boolean created = false;
        if (!this.exists(path)) {
            ((BackgroundPathAndBytesable)((ACLBackgroundPathAndBytesable)this.curator.create().withMode(CreateMode.PERSISTENT)).withACL(zkAcl)).forPath(path, null);
            created = true;
        }
        return created;
    }

    public void createRootDirRecursively(String path) throws Exception {
        this.createRootDirRecursively(path, null);
    }

    public void createRootDirRecursively(String path, List<ACL> zkAcl) throws Exception {
        String[] pathParts = path.split("/");
        Preconditions.checkArgument(pathParts.length >= 1 && pathParts[0].isEmpty(), "Invalid path: %s", path);
        StringBuilder sb = new StringBuilder();
        for (int i = 1; i < pathParts.length; ++i) {
            sb.append("/").append(pathParts[i]);
            this.create(sb.toString(), zkAcl);
        }
    }

    public boolean delete(String path) throws Exception {
        if (this.exists(path)) {
            this.curator.delete().deletingChildrenIfNeeded().forPath(path);
            return true;
        }
        return false;
    }

    public static String getNodePath(String root, String nodeName) {
        return root + "/" + nodeName;
    }

    public void safeCreate(String path, byte[] data, List<ACL> acl, CreateMode mode, List<ACL> fencingACL, String fencingNodePath) throws Exception {
        if (!this.exists(path)) {
            SafeTransaction transaction = this.createTransaction(fencingACL, fencingNodePath);
            transaction.create(path, data, acl, mode);
            transaction.commit();
        }
    }

    public void safeDelete(String path, List<ACL> fencingACL, String fencingNodePath) throws Exception {
        if (this.exists(path)) {
            SafeTransaction transaction = this.createTransaction(fencingACL, fencingNodePath);
            transaction.delete(path);
            transaction.commit();
        }
    }

    public void safeSetData(String path, byte[] data, int version, List<ACL> fencingACL, String fencingNodePath) throws Exception {
        SafeTransaction transaction = this.createTransaction(fencingACL, fencingNodePath);
        transaction.setData(path, data, version);
        transaction.commit();
    }

    public SafeTransaction createTransaction(List<ACL> fencingACL, String fencingNodePath) throws Exception {
        return new SafeTransaction(fencingACL, fencingNodePath);
    }

    public static class HadoopZookeeperFactory
    implements ZookeeperFactory {
        public static final String JAAS_CLIENT_ENTRY = "HadoopZookeeperClient";
        private final String zkPrincipal;
        private final String kerberosPrincipal;
        private final String kerberosKeytab;
        private final Boolean sslEnabled;
        private final SecurityUtil.TruststoreKeystore truststoreKeystore;

        public HadoopZookeeperFactory(String zkPrincipal) {
            this(zkPrincipal, null, null);
        }

        public HadoopZookeeperFactory(String zkPrincipal, String kerberosPrincipal, String kerberosKeytab) {
            this(zkPrincipal, kerberosPrincipal, kerberosKeytab, false, new SecurityUtil.TruststoreKeystore(new Configuration()));
        }

        public HadoopZookeeperFactory(String zkPrincipal, String kerberosPrincipal, String kerberosKeytab, boolean sslEnabled, SecurityUtil.TruststoreKeystore truststoreKeystore) {
            this.zkPrincipal = zkPrincipal;
            this.kerberosPrincipal = kerberosPrincipal;
            this.kerberosKeytab = kerberosKeytab;
            this.sslEnabled = sslEnabled;
            this.truststoreKeystore = truststoreKeystore;
        }

        public ZooKeeper newZooKeeper(String connectString, int sessionTimeout, Watcher watcher, boolean canBeReadOnly) throws Exception {
            return this.newZooKeeper(connectString, sessionTimeout, watcher, canBeReadOnly, new ZKClientConfig());
        }

        public ZooKeeper newZooKeeper(String connectString, int sessionTimeout, Watcher watcher, boolean canBeReadOnly, ZKClientConfig zkClientConfig) throws Exception {
            if (zkClientConfig == null) {
                zkClientConfig = new ZKClientConfig();
            }
            if (this.zkPrincipal != null) {
                LOG.info("Configuring zookeeper to use {} as the server principal", (Object)this.zkPrincipal);
                zkClientConfig.setProperty("zookeeper.sasl.client.username", this.zkPrincipal);
            }
            if (zkClientConfig.isSaslClientEnabled() && !this.isJaasConfigurationSet(zkClientConfig)) {
                this.setJaasConfiguration(zkClientConfig);
            }
            if (this.sslEnabled.booleanValue()) {
                SecurityUtil.setSslConfiguration(zkClientConfig, this.truststoreKeystore);
            }
            return new ZooKeeper(connectString, sessionTimeout, watcher, canBeReadOnly, zkClientConfig);
        }

        private boolean isJaasConfigurationSet(ZKClientConfig zkClientConfig) {
            String clientConfig = zkClientConfig.getProperty("zookeeper.sasl.clientconfig", "Client");
            return javax.security.auth.login.Configuration.getConfiguration().getAppConfigurationEntry(clientConfig) != null;
        }

        private void setJaasConfiguration(ZKClientConfig zkClientConfig) throws IOException {
            if (this.kerberosPrincipal == null || this.kerberosKeytab == null) {
                LOG.warn("JaasConfiguration has not been set since kerberos principal or keytab is not specified");
                return;
            }
            String principal = SecurityUtil.getServerPrincipal(this.kerberosPrincipal, "");
            JaasConfiguration jconf = new JaasConfiguration(JAAS_CLIENT_ENTRY, principal, this.kerberosKeytab);
            javax.security.auth.login.Configuration.setConfiguration(jconf);
            zkClientConfig.setProperty("zookeeper.sasl.clientconfig", JAAS_CLIENT_ENTRY);
        }
    }

    public class SafeTransaction {
        private String fencingNodePath;
        private List<CuratorOp> curatorOperations = new LinkedList<CuratorOp>();

        SafeTransaction(List<ACL> fencingACL, String fencingNodePath) throws Exception {
            this.fencingNodePath = fencingNodePath;
            this.curatorOperations.add((CuratorOp)((PathAndBytesable)((ACLPathAndBytesable)ZKCuratorManager.this.curator.transactionOp().create().withMode(CreateMode.PERSISTENT)).withACL(fencingACL)).forPath(fencingNodePath, new byte[0]));
        }

        public void commit() throws Exception {
            this.curatorOperations.add((CuratorOp)ZKCuratorManager.this.curator.transactionOp().delete().forPath(this.fencingNodePath));
            ZKCuratorManager.this.curator.transaction().forOperations(this.curatorOperations);
            this.curatorOperations.clear();
        }

        public void create(String path, byte[] data, List<ACL> acl, CreateMode mode) throws Exception {
            this.curatorOperations.add((CuratorOp)((PathAndBytesable)((ACLPathAndBytesable)ZKCuratorManager.this.curator.transactionOp().create().withMode(mode)).withACL(acl)).forPath(path, data));
        }

        public void delete(String path) throws Exception {
            this.curatorOperations.add((CuratorOp)ZKCuratorManager.this.curator.transactionOp().delete().forPath(path));
        }

        public void setData(String path, byte[] data, int version) throws Exception {
            this.curatorOperations.add((CuratorOp)((PathAndBytesable)ZKCuratorManager.this.curator.transactionOp().setData().withVersion(version)).forPath(path, data));
        }
    }
}

