/*
 * Decompiled with CFR 0.152.
 */
package io.debezium.jdbc;

import io.debezium.DebeziumException;
import io.debezium.annotation.NotThreadSafe;
import io.debezium.annotation.ThreadSafe;
import io.debezium.config.CommonConnectorConfig;
import io.debezium.config.Field;
import io.debezium.jdbc.JdbcConfiguration;
import io.debezium.pipeline.source.snapshot.incremental.ChunkQueryBuilder;
import io.debezium.pipeline.source.snapshot.incremental.DefaultChunkQueryBuilder;
import io.debezium.relational.Attribute;
import io.debezium.relational.Column;
import io.debezium.relational.ColumnEditor;
import io.debezium.relational.RelationalDatabaseConnectorConfig;
import io.debezium.relational.Table;
import io.debezium.relational.TableId;
import io.debezium.relational.Tables;
import io.debezium.spi.schema.DataCollectionId;
import io.debezium.util.BoundedConcurrentHashMap;
import io.debezium.util.Collect;
import io.debezium.util.ColumnUtils;
import io.debezium.util.Strings;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.lang.reflect.InvocationTargetException;
import java.security.KeyStore;
import java.security.KeyStoreException;
import java.security.NoSuchAlgorithmException;
import java.security.cert.CertificateException;
import java.sql.CallableStatement;
import java.sql.Connection;
import java.sql.DatabaseMetaData;
import java.sql.Driver;
import java.sql.DriverManager;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.ResultSetMetaData;
import java.sql.SQLException;
import java.sql.Statement;
import java.time.Instant;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.OptionalLong;
import java.util.Properties;
import java.util.Set;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.function.Consumer;
import java.util.function.Predicate;
import java.util.stream.Collectors;
import org.apache.kafka.connect.data.Struct;
import org.apache.kafka.connect.errors.ConnectException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@NotThreadSafe
public class JdbcConnection
implements AutoCloseable {
    private static final int WAIT_FOR_CLOSE_SECONDS = 10;
    private static final char STATEMENT_DELIMITER = ';';
    private static final String ESCAPE_CHAR = "\\";
    private static final int STATEMENT_CACHE_CAPACITY = 10000;
    private static final Logger LOGGER = LoggerFactory.getLogger(JdbcConnection.class);
    private static final int CONNECTION_VALID_CHECK_TIMEOUT_IN_SEC = 3;
    private final Map<String, PreparedStatement> statementCache = new BoundedConcurrentHashMap<String, PreparedStatement>(10000, 16, BoundedConcurrentHashMap.Eviction.LIRS, new BoundedConcurrentHashMap.EvictionListener<String, PreparedStatement>(){

        @Override
        public void onEntryEviction(Map<String, PreparedStatement> evicted) {
        }

        @Override
        public void onEntryChosenForEviction(PreparedStatement statement) {
            JdbcConnection.this.cleanupPreparedStatement(statement);
        }
    });
    private final JdbcConfiguration config;
    private final ConnectionFactory factory;
    private final Operations initialOps;
    private final String openingQuoteCharacter;
    private final String closingQuoteCharacter;
    private volatile Connection conn;
    private final int queryTimeout;

    public static ConnectionFactory patternBasedFactory(String urlPattern, Field ... variables) {
        return config -> {
            if (LOGGER.isTraceEnabled()) {
                LOGGER.trace("Config: {}", (Object)JdbcConnection.propsWithMaskedPassword(config.asProperties()));
            }
            Properties props = config.asProperties();
            Field[] varsWithDefaults = JdbcConnection.combineVariables(variables, JdbcConfiguration.HOSTNAME, JdbcConfiguration.PORT, JdbcConfiguration.USER, JdbcConfiguration.PASSWORD, JdbcConfiguration.DATABASE);
            String url = JdbcConnection.findAndReplace(urlPattern, props, varsWithDefaults);
            if (LOGGER.isTraceEnabled()) {
                LOGGER.trace("Props: {}", (Object)JdbcConnection.propsWithMaskedPassword(props));
            }
            LOGGER.trace("URL: {}", (Object)url);
            Connection conn = DriverManager.getConnection(url, props);
            if (LOGGER.isDebugEnabled()) {
                LOGGER.debug("Connected to {} with {}", (Object)url, (Object)JdbcConnection.propsWithMaskedPassword(props));
            }
            return conn;
        };
    }

    public static ConnectionFactory patternBasedFactory(String urlPattern, String driverClassName, ClassLoader classloader, Field ... variables) {
        return config -> {
            if (LOGGER.isTraceEnabled()) {
                LOGGER.trace("Config: {}", (Object)JdbcConnection.propsWithMaskedPassword(config.asProperties()));
            }
            Properties props = config.asProperties();
            Field[] varsWithDefaults = JdbcConnection.combineVariables(variables, JdbcConfiguration.HOSTNAME, JdbcConfiguration.PORT, JdbcConfiguration.USER, JdbcConfiguration.PASSWORD, JdbcConfiguration.DATABASE);
            String url = JdbcConnection.findAndReplace(urlPattern, props, varsWithDefaults);
            if (LOGGER.isTraceEnabled()) {
                LOGGER.trace("Props: {}", (Object)JdbcConnection.propsWithMaskedPassword(props));
            }
            LOGGER.trace("URL: {}", (Object)url);
            Connection conn = null;
            try {
                ClassLoader driverClassLoader = classloader;
                if (driverClassLoader == null) {
                    driverClassLoader = JdbcConnection.class.getClassLoader();
                }
                Class<?> driverClazz = Class.forName(driverClassName, true, driverClassLoader);
                Driver driver = (Driver)driverClazz.getDeclaredConstructor(new Class[0]).newInstance(new Object[0]);
                conn = driver.connect(url, props);
            }
            catch (ClassNotFoundException | IllegalAccessException | InstantiationException | NoSuchMethodException | InvocationTargetException e) {
                throw new SQLException(e);
            }
            if (LOGGER.isDebugEnabled()) {
                LOGGER.debug("Connected to {} with {}", (Object)url, (Object)JdbcConnection.propsWithMaskedPassword(props));
            }
            return conn;
        };
    }

    private static Properties propsWithMaskedPassword(Properties props) {
        Properties filtered = new Properties();
        filtered.putAll((Map<?, ?>)props);
        if (props.containsKey(JdbcConfiguration.PASSWORD.name())) {
            filtered.put(JdbcConfiguration.PASSWORD.name(), "***");
        }
        return filtered;
    }

    public Optional<Instant> getCurrentTimestamp() throws SQLException {
        return this.queryAndMap("SELECT CURRENT_TIMESTAMP", rs -> rs.next() ? Optional.of(rs.getTimestamp(1).toInstant()) : Optional.empty());
    }

    private static Field[] combineVariables(Field[] overriddenVariables, Field ... defaultVariables) {
        HashMap<String, Field> fields = new HashMap<String, Field>();
        if (defaultVariables != null) {
            for (Field variable : defaultVariables) {
                fields.put(variable.name(), variable);
            }
        }
        if (overriddenVariables != null) {
            for (Field variable : overriddenVariables) {
                fields.put(variable.name(), variable);
            }
        }
        return fields.values().toArray(new Field[0]);
    }

    private static String findAndReplace(String url, Properties props, Field ... variables) {
        for (Field field : variables) {
            if (field == null) continue;
            url = JdbcConnection.findAndReplace(url, field.name(), props, field.defaultValueAsString());
        }
        for (Object e : new HashSet<Object>(props.keySet())) {
            if (e == null) continue;
            url = JdbcConnection.findAndReplace(url, e.toString(), props, null);
        }
        return url;
    }

    private static String findAndReplace(String url, String name, Properties props, String defaultValue) {
        if (name != null && url.contains("${" + name + "}")) {
            String value = props.getProperty(name);
            if (value != null) {
                props.remove(name);
            }
            if (value == null) {
                value = defaultValue;
            }
            if (value != null) {
                url = url.replaceAll("\\$\\{" + name + "\\}", value);
            }
        }
        return url;
    }

    public JdbcConnection(JdbcConfiguration config, ConnectionFactory connectionFactory, String openingQuoteCharacter, String closingQuoteCharacter) {
        this(config, connectionFactory, null, openingQuoteCharacter, closingQuoteCharacter);
    }

    protected JdbcConnection(JdbcConfiguration config, ConnectionFactory connectionFactory, Operations initialOperations, String openingQuotingChar, String closingQuotingChar) {
        this.config = config;
        this.factory = new ConnectionFactoryDecorator(connectionFactory);
        this.initialOps = initialOperations;
        this.openingQuoteCharacter = openingQuotingChar;
        this.closingQuoteCharacter = closingQuotingChar;
        this.conn = null;
        this.queryTimeout = (int)config.getQueryTimeout().toSeconds();
    }

    public JdbcConfiguration config() {
        return this.config;
    }

    public JdbcConnection setAutoCommit(boolean autoCommit) throws SQLException {
        this.connection().setAutoCommit(autoCommit);
        return this;
    }

    public JdbcConnection commit() throws SQLException {
        Connection conn = this.connection();
        if (!conn.getAutoCommit()) {
            conn.commit();
        }
        return this;
    }

    public synchronized JdbcConnection rollback() throws SQLException {
        if (!this.isConnected()) {
            return this;
        }
        Connection conn = this.connection();
        if (!conn.getAutoCommit()) {
            conn.rollback();
        }
        return this;
    }

    public JdbcConnection connect() throws SQLException {
        this.connection();
        return this;
    }

    public void reconnect() throws SQLException {
        this.establishConnection();
    }

    public JdbcConnection execute(String ... sqlStatements) throws SQLException {
        return this.execute((Statement statement) -> {
            for (String sqlStatement : sqlStatements) {
                if (sqlStatement == null) continue;
                if (LOGGER.isTraceEnabled()) {
                    LOGGER.trace("executing '{}'", (Object)sqlStatement);
                }
                statement.execute(sqlStatement);
            }
        });
    }

    public JdbcConnection execute(Operations operations) throws SQLException {
        Connection conn = this.connection();
        try (Statement statement = conn.createStatement();){
            statement.setQueryTimeout(this.queryTimeout);
            if (LOGGER.isTraceEnabled()) {
                LOGGER.trace("Executing query with {}s timeout", (Object)this.queryTimeout);
            }
            operations.apply(statement);
            this.commit();
        }
        return this;
    }

    public JdbcConnection query(String query, ResultSetConsumer resultConsumer) throws SQLException {
        return this.query(query, Connection::createStatement, resultConsumer);
    }

    public <T> T queryAndMap(String query, ResultSetMapper<T> mapper) throws SQLException {
        return this.queryAndMap(query, Connection::createStatement, mapper);
    }

    public JdbcConnection call(String sql, CallPreparer callPreparer, ResultSetConsumer resultSetConsumer) throws SQLException {
        Connection conn = this.connection();
        try (CallableStatement callableStatement = conn.prepareCall(sql);){
            if (callPreparer != null) {
                callPreparer.accept(callableStatement);
            }
            try (ResultSet rs = callableStatement.executeQuery();){
                if (resultSetConsumer != null) {
                    resultSetConsumer.accept(rs);
                }
            }
        }
        return this;
    }

    public JdbcConnection query(String query, StatementFactory statementFactory, ResultSetConsumer resultConsumer) throws SQLException {
        Connection conn = this.connection();
        try (Statement statement = statementFactory.createStatement(conn);){
            statement.setQueryTimeout(this.queryTimeout);
            if (LOGGER.isTraceEnabled()) {
                LOGGER.trace("running '{}' with {}s timeout", (Object)query, (Object)this.queryTimeout);
            }
            try (ResultSet resultSet = statement.executeQuery(query);){
                if (resultConsumer != null) {
                    resultConsumer.accept(resultSet);
                }
            }
        }
        return this;
    }

    public JdbcConnection prepareQuery(String[] multiQuery, StatementPreparer preparer, BlockingMultiResultSetConsumer resultConsumer) throws SQLException, InterruptedException {
        Object[] preparers = new StatementPreparer[multiQuery.length];
        Arrays.fill(preparers, preparer);
        return this.prepareQuery(multiQuery, (StatementPreparer[])preparers, resultConsumer);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public JdbcConnection prepareQuery(String[] multiQuery, StatementPreparer[] preparers, BlockingMultiResultSetConsumer resultConsumer) throws SQLException, InterruptedException {
        ResultSet[] resultSets = new ResultSet[multiQuery.length];
        PreparedStatement[] preparedStatements = new PreparedStatement[multiQuery.length];
        try {
            for (int i = 0; i < multiQuery.length; ++i) {
                PreparedStatement statement;
                String query = multiQuery[i];
                if (LOGGER.isTraceEnabled()) {
                    LOGGER.trace("running '{}'", (Object)query);
                }
                preparedStatements[i] = statement = this.createPreparedStatement(query);
                preparers[i].accept(statement);
                resultSets[i] = statement.executeQuery();
            }
            if (resultConsumer != null) {
                resultConsumer.accept(resultSets);
            }
        }
        finally {
            for (ResultSet rs : resultSets) {
                if (rs == null) continue;
                try {
                    rs.close();
                }
                catch (Exception exception) {}
            }
        }
        return this;
    }

    public <T> T queryAndMap(String query, StatementFactory statementFactory, ResultSetMapper<T> mapper) throws SQLException {
        Objects.requireNonNull(mapper, "Mapper must be provided");
        Connection conn = this.connection();
        try (Statement statement = statementFactory.createStatement(conn);){
            T t;
            block13: {
                statement.setQueryTimeout(this.queryTimeout);
                if (LOGGER.isTraceEnabled()) {
                    LOGGER.trace("running '{}' with {}s timeout", (Object)query, (Object)this.queryTimeout);
                }
                ResultSet resultSet = statement.executeQuery(query);
                try {
                    t = mapper.apply(resultSet);
                    if (resultSet == null) break block13;
                }
                catch (Throwable throwable) {
                    if (resultSet != null) {
                        try {
                            resultSet.close();
                        }
                        catch (Throwable throwable2) {
                            throwable.addSuppressed(throwable2);
                        }
                    }
                    throw throwable;
                }
                resultSet.close();
            }
            return t;
        }
    }

    public JdbcConnection queryWithBlockingConsumer(String query, StatementFactory statementFactory, BlockingResultSetConsumer resultConsumer) throws SQLException, InterruptedException {
        Connection conn = this.connection();
        try (Statement statement = statementFactory.createStatement(conn);){
            statement.setQueryTimeout(this.queryTimeout);
            if (LOGGER.isTraceEnabled()) {
                LOGGER.trace("running '{}' with {}s timeout", (Object)query, (Object)this.queryTimeout);
            }
            try (ResultSet resultSet = statement.executeQuery(query);){
                if (resultConsumer != null) {
                    resultConsumer.accept(resultSet);
                }
            }
        }
        return this;
    }

    public JdbcConnection prepareQueryWithBlockingConsumer(String preparedQueryString, StatementPreparer preparer, BlockingResultSetConsumer resultConsumer) throws SQLException, InterruptedException {
        PreparedStatement statement = this.createPreparedStatement(preparedQueryString);
        preparer.accept(statement);
        statement.setQueryTimeout(this.queryTimeout);
        if (LOGGER.isTraceEnabled()) {
            LOGGER.trace("Executing '{}' with {}s timeout", (Object)preparedQueryString, (Object)this.queryTimeout);
        }
        try (ResultSet resultSet = statement.executeQuery();){
            if (resultConsumer != null) {
                resultConsumer.accept(resultSet);
            }
        }
        return this;
    }

    public JdbcConnection prepareQuery(String preparedQueryString) throws SQLException {
        PreparedStatement statement = this.createPreparedStatement(preparedQueryString);
        statement.executeQuery();
        return this;
    }

    public JdbcConnection prepareQuery(String preparedQueryString, StatementPreparer preparer, ResultSetConsumer resultConsumer) throws SQLException {
        PreparedStatement statement = this.createPreparedStatement(preparedQueryString);
        preparer.accept(statement);
        try (ResultSet resultSet = statement.executeQuery();){
            if (resultConsumer != null) {
                resultConsumer.accept(resultSet);
            }
        }
        return this;
    }

    public <T> T prepareQueryAndMap(String preparedQueryString, StatementPreparer preparer, ResultSetMapper<T> mapper) throws SQLException {
        Objects.requireNonNull(mapper, "Mapper must be provided");
        PreparedStatement statement = this.createPreparedStatement(preparedQueryString);
        preparer.accept(statement);
        try (ResultSet resultSet = statement.executeQuery();){
            T t = mapper.apply(resultSet);
            return t;
        }
    }

    public JdbcConnection prepareUpdate(String stmt, StatementPreparer preparer) throws SQLException {
        PreparedStatement statement = this.createPreparedStatement(stmt);
        if (preparer != null) {
            preparer.accept(statement);
        }
        statement.setQueryTimeout(this.queryTimeout);
        if (LOGGER.isTraceEnabled()) {
            LOGGER.trace("Executing statement '{}' with {}s timeout", (Object)stmt, (Object)this.queryTimeout);
        }
        statement.execute();
        return this;
    }

    public JdbcConnection prepareQuery(String preparedQueryString, List<?> parameters, ParameterResultSetConsumer resultConsumer) throws SQLException {
        PreparedStatement statement = this.createPreparedStatement(preparedQueryString);
        int index = 1;
        for (Object parameter : parameters) {
            statement.setObject(index++, parameter);
        }
        try (ResultSet resultSet = statement.executeQuery();){
            if (resultConsumer != null) {
                resultConsumer.accept(parameters, resultSet);
            }
        }
        return this;
    }

    public void print(ResultSet resultSet) {
        this.print(resultSet, System.out::println);
    }

    public void print(ResultSet resultSet, Consumer<String> lines) {
        try {
            int i;
            ResultSetMetaData rsmd = resultSet.getMetaData();
            int columnCount = rsmd.getColumnCount();
            int[] columnSizes = this.findMaxLength(resultSet);
            lines.accept(this.delimiter(columnCount, columnSizes));
            StringBuilder sb = new StringBuilder();
            for (i = 1; i <= columnCount; ++i) {
                if (i > 1) {
                    sb.append(" | ");
                }
                sb.append(Strings.setLength(rsmd.getColumnLabel(i), columnSizes[i], ' '));
            }
            lines.accept(sb.toString());
            sb.setLength(0);
            lines.accept(this.delimiter(columnCount, columnSizes));
            while (resultSet.next()) {
                sb.setLength(0);
                for (i = 1; i <= columnCount; ++i) {
                    if (i > 1) {
                        sb.append(" | ");
                    }
                    sb.append(Strings.setLength(resultSet.getString(i), columnSizes[i], ' '));
                }
                lines.accept(sb.toString());
                sb.setLength(0);
            }
            lines.accept(this.delimiter(columnCount, columnSizes));
        }
        catch (SQLException e) {
            throw new RuntimeException(e);
        }
    }

    private String delimiter(int columnCount, int[] columnSizes) {
        StringBuilder sb = new StringBuilder();
        for (int i = 1; i <= columnCount; ++i) {
            if (i > 1) {
                sb.append("---");
            }
            sb.append(Strings.createString('-', columnSizes[i]));
        }
        return sb.toString();
    }

    private int[] findMaxLength(ResultSet resultSet) throws SQLException {
        int i;
        ResultSetMetaData rsmd = resultSet.getMetaData();
        int columnCount = rsmd.getColumnCount();
        int[] columnSizes = new int[columnCount + 1];
        for (i = 1; i <= columnCount; ++i) {
            columnSizes[i] = Math.max(columnSizes[i], rsmd.getColumnLabel(i).length());
        }
        while (resultSet.next()) {
            for (i = 1; i <= columnCount; ++i) {
                String value = resultSet.getString(i);
                if (value == null) continue;
                columnSizes[i] = Math.max(columnSizes[i], value.length());
            }
        }
        resultSet.beforeFirst();
        return columnSizes;
    }

    public synchronized boolean isConnected() throws SQLException {
        if (this.conn == null) {
            return false;
        }
        return !this.conn.isClosed();
    }

    public synchronized boolean isValid() throws SQLException {
        return this.isConnected() && this.conn.isValid(3);
    }

    public synchronized Connection connection() throws SQLException {
        return this.connection(true);
    }

    public synchronized Connection connection(boolean executeOnConnect) throws SQLException {
        if (!this.isConnected()) {
            String statements;
            this.establishConnection();
            if (this.initialOps != null) {
                this.execute(this.initialOps);
            }
            if ((statements = this.config.getString(JdbcConfiguration.ON_CONNECT_STATEMENTS)) != null && executeOnConnect) {
                List<String> splitStatements = this.parseSqlStatementString(statements);
                this.execute(splitStatements.toArray(new String[splitStatements.size()]));
            }
        }
        return this.conn;
    }

    private void establishConnection() throws SQLException {
        this.conn = this.factory.connect(JdbcConfiguration.adapt(this.config));
        if (!this.isConnected()) {
            throw new SQLException("Unable to obtain a JDBC connection");
        }
    }

    protected List<String> parseSqlStatementString(String statements) {
        ArrayList<String> splitStatements = new ArrayList<String>();
        char[] statementsChars = statements.toCharArray();
        StringBuilder activeStatement = new StringBuilder();
        for (int i = 0; i < statementsChars.length; ++i) {
            if (statementsChars[i] == ';') {
                if (i == statementsChars.length - 1) continue;
                if (statementsChars[i + 1] == ';') {
                    activeStatement.append(';');
                    ++i;
                    continue;
                }
                String trimmedStatement = activeStatement.toString().trim();
                if (!trimmedStatement.isEmpty()) {
                    splitStatements.add(trimmedStatement);
                }
                activeStatement = new StringBuilder();
                continue;
            }
            activeStatement.append(statementsChars[i]);
        }
        String trimmedStatement = activeStatement.toString().trim();
        if (!trimmedStatement.isEmpty()) {
            splitStatements.add(trimmedStatement);
        }
        return splitStatements;
    }

    @Override
    public synchronized void close() throws SQLException {
        if (this.conn != null) {
            try {
                this.statementCache.values().forEach(this::cleanupPreparedStatement);
                this.statementCache.clear();
                LOGGER.trace("Closing database connection");
                this.doClose();
            }
            finally {
                this.conn = null;
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void doClose() throws SQLException {
        ExecutorService executor = Executors.newSingleThreadExecutor();
        Future<Object> futureClose = executor.submit(() -> {
            this.conn.close();
            LOGGER.info("Connection gracefully closed");
            return null;
        });
        try {
            futureClose.get(10L, TimeUnit.SECONDS);
        }
        catch (ExecutionException e) {
            if (e.getCause() instanceof SQLException) {
                throw (SQLException)e.getCause();
            }
            if (e.getCause() instanceof RuntimeException) {
                throw (RuntimeException)e.getCause();
            }
            throw new DebeziumException(e.getCause());
        }
        catch (InterruptedException | TimeoutException e) {
            LOGGER.warn("Failed to close database connection by calling close(), attempting abort()");
            this.conn.abort(Runnable::run);
        }
        finally {
            executor.shutdownNow();
        }
    }

    public Set<String> readAllCatalogNames() throws SQLException {
        HashSet<String> catalogs = new HashSet<String>();
        DatabaseMetaData metadata = this.connection().getMetaData();
        try (ResultSet rs = metadata.getCatalogs();){
            while (rs.next()) {
                String catalogName = rs.getString(1);
                catalogs.add(catalogName);
            }
        }
        return catalogs;
    }

    public Set<String> readAllSchemaNames(Predicate<String> filter) throws SQLException {
        HashSet<String> schemas = new HashSet<String>();
        DatabaseMetaData metadata = this.connection().getMetaData();
        try (ResultSet rs = metadata.getSchemas();){
            while (rs.next()) {
                String schema = rs.getString(1);
                if (filter == null || !filter.test(schema)) continue;
                schemas.add(schema);
            }
        }
        return schemas;
    }

    public String[] tableTypes() throws SQLException {
        ArrayList<String> types = new ArrayList<String>();
        DatabaseMetaData metadata = this.connection().getMetaData();
        try (ResultSet rs = metadata.getTableTypes();){
            while (rs.next()) {
                String tableType = rs.getString(1);
                if (tableType == null) continue;
                types.add(tableType);
            }
        }
        return types.toArray(new String[types.size()]);
    }

    public Set<TableId> readAllTableNames(String[] tableTypes) throws SQLException {
        return this.readTableNames(null, null, null, tableTypes);
    }

    public Set<TableId> readTableNames(String databaseCatalog, String schemaNamePattern, String tableNamePattern, String[] tableTypes) throws SQLException {
        if (tableNamePattern == null) {
            tableNamePattern = "%";
        }
        HashSet<TableId> tableIds = new HashSet<TableId>();
        DatabaseMetaData metadata = this.connection().getMetaData();
        try (ResultSet rs = metadata.getTables(databaseCatalog, schemaNamePattern, tableNamePattern, tableTypes);){
            while (rs.next()) {
                String catalogName = rs.getString(1);
                String schemaName = rs.getString(2);
                String tableName = rs.getString(3);
                TableId tableId = new TableId(catalogName, schemaName, tableName);
                tableIds.add(tableId);
            }
        }
        return tableIds;
    }

    public String connectionString(String urlPattern) {
        Properties props = this.config.asProperties();
        return JdbcConnection.findAndReplace(urlPattern, props, JdbcConfiguration.DATABASE, JdbcConfiguration.HOSTNAME, JdbcConfiguration.PORT, JdbcConfiguration.USER, JdbcConfiguration.PASSWORD);
    }

    public String username() {
        return this.config.getString(JdbcConfiguration.USER);
    }

    public String database() {
        return this.config.getString(JdbcConfiguration.DATABASE);
    }

    protected int resolveNativeType(String typeName) {
        return -1;
    }

    protected int resolveJdbcType(int metadataJdbcType, int nativeType) {
        return metadataJdbcType;
    }

    public void readSchema(Tables tables, String databaseCatalog, String schemaNamePattern, Tables.TableFilter tableFilter, Tables.ColumnNameFilter columnFilter, boolean removeTablesNotFoundInJdbc) throws SQLException {
        HashSet<TableId> tableIdsBefore = new HashSet<TableId>(tables.tableIds());
        DatabaseMetaData metadata = this.connection().getMetaData();
        HashSet<TableId> viewIds = new HashSet<TableId>();
        HashSet<TableId> tableIds = new HashSet<TableId>();
        HashMap<TableId, List<Attribute>> attributesByTable = new HashMap<TableId, List<Attribute>>();
        int totalTables = 0;
        try (ResultSet rs = metadata.getTables(databaseCatalog, schemaNamePattern, null, this.supportedTableTypes());){
            while (rs.next()) {
                TableId tableId;
                String catalogName = this.resolveCatalogName(rs.getString(1));
                String string = rs.getString(2);
                String tableName = rs.getString(3);
                String tableType = rs.getString(4);
                if (this.isTableType(tableType)) {
                    ++totalTables;
                    tableId = new TableId(catalogName, string, tableName);
                    if (tableFilter != null && !tableFilter.isIncluded(tableId)) continue;
                    tableIds.add(tableId);
                    attributesByTable.putAll(this.getAttributeDetails(tableId, tableType));
                    continue;
                }
                tableId = new TableId(catalogName, string, tableName);
                viewIds.add(tableId);
            }
        }
        LOGGER.debug("{} table(s) will be scanned", (Object)tableIds.size());
        Map<Object, Object> columnsByTable = new HashMap();
        if (totalTables == tableIds.size() || this.config.getBoolean(RelationalDatabaseConnectorConfig.SNAPSHOT_FULL_COLUMN_SCAN_FORCE)) {
            columnsByTable = this.getColumnsDetails(databaseCatalog, schemaNamePattern, null, tableFilter, columnFilter, metadata, viewIds);
        } else {
            for (TableId tableId : tableIds) {
                LOGGER.debug("Retrieving columns of table {}", (Object)tableId);
                Map<TableId, List<Column>> cols = this.getColumnsDetails(databaseCatalog, schemaNamePattern, tableId.table(), tableFilter, columnFilter, metadata, viewIds);
                columnsByTable.putAll(cols);
            }
        }
        for (Map.Entry entry : columnsByTable.entrySet()) {
            List<String> pkColumnNames = this.readPrimaryKeyOrUniqueIndexNames(metadata, (TableId)entry.getKey());
            List columns = (List)entry.getValue();
            Collections.sort(columns);
            String defaultCharsetName = null;
            List<Attribute> attributes = attributesByTable.getOrDefault(entry.getKey(), Collections.emptyList());
            tables.overwriteTable((TableId)entry.getKey(), columns, pkColumnNames, defaultCharsetName, attributes);
        }
        if (removeTablesNotFoundInJdbc) {
            tableIdsBefore.removeAll(columnsByTable.keySet());
            tableIdsBefore.forEach(tables::removeTable);
        }
    }

    protected String[] supportedTableTypes() {
        return new String[]{"VIEW", "MATERIALIZED VIEW", "TABLE"};
    }

    protected boolean isTableType(String tableType) {
        return "TABLE".equals(tableType);
    }

    protected String resolveCatalogName(String catalogName) {
        return catalogName;
    }

    protected String escapeEscapeSequence(String str) {
        return str.replace(ESCAPE_CHAR, ESCAPE_CHAR.concat(ESCAPE_CHAR));
    }

    protected Map<TableId, List<Column>> getColumnsDetails(String databaseCatalog, String schemaNamePattern, String tableName, Tables.TableFilter tableFilter, Tables.ColumnNameFilter columnFilter, DatabaseMetaData metadata, Set<TableId> viewIds) throws SQLException {
        HashMap<TableId, List<Column>> columnsByTable = new HashMap<TableId, List<Column>>();
        if (tableName != null && tableName.contains(ESCAPE_CHAR)) {
            tableName = this.escapeEscapeSequence(tableName);
        }
        try (ResultSet columnMetadata = metadata.getColumns(databaseCatalog, schemaNamePattern, tableName, null);){
            while (columnMetadata.next()) {
                String metaTableName;
                String schemaName;
                String catalogName = this.resolveCatalogName(columnMetadata.getString(1));
                TableId tableId = new TableId(catalogName, schemaName = columnMetadata.getString(2), metaTableName = columnMetadata.getString(3));
                if (viewIds.contains(tableId) || tableFilter != null && !tableFilter.isIncluded(tableId)) continue;
                this.readTableColumn(columnMetadata, tableId, columnFilter).ifPresent(column -> columnsByTable.computeIfAbsent(tableId, t -> new ArrayList()).add(column.create()));
            }
        }
        return columnsByTable;
    }

    protected Map<TableId, List<Attribute>> getAttributeDetails(TableId tableId, String tableType) {
        return Collections.emptyMap();
    }

    protected Optional<ColumnEditor> readTableColumn(ResultSet columnMetadata, TableId tableId, Tables.ColumnNameFilter columnFilter) throws SQLException {
        String defaultValue = columnMetadata.getString(13);
        String columnName = columnMetadata.getString(4);
        if (columnFilter == null || columnFilter.matches(tableId.catalog(), tableId.schema(), tableId.table(), columnName)) {
            ColumnEditor column = Column.editor().name(columnName);
            column.type(columnMetadata.getString(6));
            column.length(columnMetadata.getInt(7));
            if (columnMetadata.getObject(9) != null) {
                column.scale(columnMetadata.getInt(9));
            }
            column.optional(JdbcConnection.isNullable(columnMetadata.getInt(11)));
            column.position(columnMetadata.getInt(17));
            column.autoIncremented("YES".equalsIgnoreCase(columnMetadata.getString(23)));
            String autogenerated = null;
            try {
                autogenerated = columnMetadata.getString(24);
            }
            catch (SQLException sQLException) {
                // empty catch block
            }
            column.generated("YES".equalsIgnoreCase(autogenerated));
            column.nativeType(this.resolveNativeType(column.typeName()));
            column.jdbcType(this.resolveJdbcType(columnMetadata.getInt(5), column.nativeType()));
            column = this.overrideColumn(column);
            if (defaultValue != null) {
                column.defaultValueExpression(defaultValue);
            }
            return Optional.of(column);
        }
        return Optional.empty();
    }

    protected ColumnEditor overrideColumn(ColumnEditor column) {
        return column;
    }

    public List<String> readPrimaryKeyNames(DatabaseMetaData metadata, TableId id) throws SQLException {
        ArrayList<String> pkColumnNames = new ArrayList<String>();
        try (ResultSet rs = metadata.getPrimaryKeys(id.catalog(), id.schema(), id.table());){
            while (rs.next()) {
                String columnName = rs.getString(4);
                int columnIndex = rs.getInt(5);
                Collect.set(pkColumnNames, columnIndex - 1, columnName, null);
            }
        }
        return pkColumnNames;
    }

    public List<String> readTableUniqueIndices(DatabaseMetaData metadata, TableId id) throws SQLException {
        ArrayList<String> uniqueIndexColumnNames = new ArrayList<String>();
        HashSet<String> excludedIndexNames = new HashSet<String>();
        try (ResultSet rs = metadata.getIndexInfo(id.catalog(), id.schema(), id.table(), true, true);){
            String firstIndexName = null;
            while (rs.next()) {
                String indexName = rs.getString(6);
                String columnName = rs.getString(9);
                int columnIndex = rs.getInt(8);
                if (indexName == null || excludedIndexNames.contains(indexName)) continue;
                boolean indexIncluded = this.isTableUniqueIndexIncluded(indexName, columnName);
                if (!indexIncluded) {
                    excludedIndexNames.add(indexName);
                    if (firstIndexName == null || indexName.equals(firstIndexName)) {
                        firstIndexName = null;
                        uniqueIndexColumnNames.clear();
                        continue;
                    }
                }
                if (firstIndexName == null) {
                    firstIndexName = indexName;
                }
                if (!indexName.equals(firstIndexName)) {
                    ArrayList<String> arrayList = uniqueIndexColumnNames;
                    return arrayList;
                }
                if (columnName == null) continue;
                Collect.set(uniqueIndexColumnNames, columnIndex - 1, columnName, null);
            }
        }
        return uniqueIndexColumnNames;
    }

    protected List<String> readPrimaryKeyOrUniqueIndexNames(DatabaseMetaData metadata, TableId id) throws SQLException {
        List<String> pkColumnNames = this.readPrimaryKeyNames(metadata, id);
        return pkColumnNames.isEmpty() ? this.readTableUniqueIndices(metadata, id) : pkColumnNames;
    }

    protected boolean isTableUniqueIndexIncluded(String indexName, String columnName) {
        return true;
    }

    private void cleanupPreparedStatement(PreparedStatement statement) {
        LOGGER.trace("Closing prepared statement '{}' removed from cache", (Object)statement);
        try {
            statement.close();
        }
        catch (Exception e) {
            LOGGER.info("Exception while closing a prepared statement removed from cache", e);
        }
    }

    private PreparedStatement createPreparedStatement(String preparedQueryString) {
        return this.statementCache.computeIfAbsent(preparedQueryString, query -> {
            try {
                LOGGER.trace("Inserting prepared statement '{}' removed from the cache", query);
                PreparedStatement preparedStatement = this.connection().prepareStatement((String)query);
                preparedStatement.setQueryTimeout(this.queryTimeout);
                if (LOGGER.isTraceEnabled()) {
                    LOGGER.trace("PreparedStatement '{}' with {}s timeout", (Object)preparedQueryString, (Object)this.queryTimeout);
                }
                return preparedStatement;
            }
            catch (SQLException e) {
                throw new ConnectException(e);
            }
        });
    }

    public JdbcConnection executeWithoutCommitting(String ... statements) throws SQLException {
        Connection conn = this.connection();
        if (conn.getAutoCommit()) {
            throw new DebeziumException("Cannot execute without committing because auto-commit is enabled");
        }
        try (Statement statement = conn.createStatement();){
            statement.setQueryTimeout(this.queryTimeout);
            for (String stmt : statements) {
                if (LOGGER.isTraceEnabled()) {
                    LOGGER.trace("Executing statement '{}' with {}s timeout", (Object)stmt, (Object)this.queryTimeout);
                }
                statement.execute(stmt);
            }
        }
        return this;
    }

    protected static boolean isNullable(int jdbcNullable) {
        return jdbcNullable == 1 || jdbcNullable == 2;
    }

    public <T> ResultSetMapper<T> singleResultMapper(ResultSetExtractor<T> extractor, String error) throws SQLException {
        return rs -> {
            if (rs.next()) {
                Object ret = extractor.apply(rs);
                if (!rs.next()) {
                    return ret;
                }
            }
            throw new IllegalStateException(error);
        };
    }

    public static <T> T querySingleValue(Connection connection, String queryString, StatementPreparer preparer, ResultSetExtractor<T> extractor) throws SQLException {
        PreparedStatement preparedStatement = connection.prepareStatement(queryString);
        preparer.accept(preparedStatement);
        try (ResultSet resultSet = preparedStatement.executeQuery();){
            if (resultSet.next()) {
                T result = extractor.apply(resultSet);
                if (!resultSet.next()) {
                    T t = result;
                    return t;
                }
            }
            throw new IllegalStateException("Exactly one result expected.");
        }
    }

    public <T extends DataCollectionId> ChunkQueryBuilder<T> chunkQueryBuilder(RelationalDatabaseConnectorConfig connectorConfig) {
        return new DefaultChunkQueryBuilder(connectorConfig, this);
    }

    public String buildSelectWithRowLimits(TableId tableId, int limit, String projection, Optional<String> condition, Optional<String> additionalCondition, String orderBy) {
        StringBuilder sql = new StringBuilder("SELECT ");
        sql.append(projection).append(" FROM ");
        sql.append(this.quotedTableIdString(tableId));
        if (condition.isPresent()) {
            sql.append(" WHERE ").append(condition.get());
            if (additionalCondition.isPresent()) {
                sql.append(" AND ");
                sql.append(additionalCondition.get());
            }
        } else if (additionalCondition.isPresent()) {
            sql.append(" WHERE ");
            sql.append(additionalCondition.get());
        }
        sql.append(" ORDER BY ").append(orderBy).append(" LIMIT ").append(limit);
        return sql.toString();
    }

    public Optional<Boolean> nullsSortLast() {
        return Optional.empty();
    }

    public Statement readTableStatement(CommonConnectorConfig connectorConfig, OptionalLong tableSize) throws SQLException {
        int fetchSize = connectorConfig.getSnapshotFetchSize();
        Statement statement = this.connection().createStatement();
        statement.setQueryTimeout(this.queryTimeout);
        if (LOGGER.isTraceEnabled()) {
            LOGGER.trace("Created Statement with {}s timeout", (Object)this.queryTimeout);
        }
        statement.setFetchSize(fetchSize);
        return statement;
    }

    public PreparedStatement readTablePreparedStatement(CommonConnectorConfig connectorConfig, String sql, OptionalLong tableSize) throws SQLException {
        int fetchSize = connectorConfig.getSnapshotFetchSize();
        PreparedStatement statement = this.connection().prepareStatement(sql);
        statement.setQueryTimeout(this.queryTimeout);
        if (LOGGER.isTraceEnabled()) {
            LOGGER.trace("PreparedStatement '{}' with {}s timeout", (Object)sql, (Object)this.queryTimeout);
        }
        statement.setFetchSize(fetchSize);
        return statement;
    }

    public Object getColumnValue(ResultSet rs, int columnIndex, Column column, Table table) throws SQLException {
        return rs.getObject(columnIndex);
    }

    public void setQueryColumnValue(PreparedStatement statement, Column column, int pos, Object value) throws SQLException {
        statement.setObject(pos, value);
    }

    public Object[] rowToArray(Table table, ResultSet rs, ColumnUtils.ColumnArray columnArray) throws SQLException {
        Object[] row = new Object[columnArray.getGreatestColumnPosition()];
        for (int i = 0; i < columnArray.getColumns().length; ++i) {
            row[columnArray.getColumns()[i].position() - 1] = this.getColumnValue(rs, i + 1, columnArray.getColumns()[i], table);
        }
        return row;
    }

    public String quotedTableIdString(TableId tableId) {
        return tableId.toDoubleQuotedString();
    }

    public String quotedColumnIdString(String columnName) {
        return this.openingQuoteCharacter + columnName + this.closingQuoteCharacter;
    }

    public KeyStore loadKeyStore(String filePath, char[] passwordArray) {
        KeyStore keyStore;
        FileInputStream in = new FileInputStream(filePath);
        try {
            KeyStore ks = KeyStore.getInstance("JKS");
            ks.load(in, passwordArray);
            keyStore = ks;
        }
        catch (Throwable throwable) {
            try {
                try {
                    ((InputStream)in).close();
                }
                catch (Throwable throwable2) {
                    throwable.addSuppressed(throwable2);
                }
                throw throwable;
            }
            catch (IOException | KeyStoreException | NoSuchAlgorithmException | CertificateException e) {
                throw new DebeziumException("Error loading keystore", e);
            }
        }
        ((InputStream)in).close();
        return keyStore;
    }

    public TableId createTableId(String databaseName, String schemaName, String tableName) {
        return new TableId(databaseName, schemaName, tableName);
    }

    public String getQualifiedTableName(TableId tableId) {
        return tableId.schema() + "." + tableId.table();
    }

    public Map<String, Object> reselectColumns(Table table, List<String> columns, List<String> keyColumns, List<Object> keyValues, Struct source) throws SQLException {
        String query = String.format("SELECT %s FROM %s WHERE %s", columns.stream().map(this::quotedColumnIdString).collect(Collectors.joining(",")), this.quotedTableIdString(table.id()), keyColumns.stream().map(key -> key + "=?").collect(Collectors.joining(" AND ")));
        return this.reselectColumns(query, table.id(), columns, keyValues);
    }

    protected Map<String, Object> reselectColumns(String query, TableId tableId, List<String> columns, List<Object> bindValues) throws SQLException {
        HashMap<String, Object> results = new HashMap<String, Object>();
        this.prepareQuery(query, bindValues, (List<?> params, ResultSet rs) -> {
            if (!rs.next()) {
                LOGGER.warn("No data found for re-selection on table {}.", (Object)tableId);
                return;
            }
            for (String columnName : columns) {
                results.put(columnName, rs.getObject(columnName));
            }
            if (rs.next()) {
                LOGGER.warn("Re-selection detected multiple rows for the same key in table {}, using first.", (Object)tableId);
            }
        });
        return results;
    }

    @FunctionalInterface
    @ThreadSafe
    public static interface ConnectionFactory {
        public Connection connect(JdbcConfiguration var1) throws SQLException;
    }

    public static interface ResultSetMapper<T> {
        public T apply(ResultSet var1) throws SQLException;
    }

    @FunctionalInterface
    public static interface Operations {
        public void apply(Statement var1) throws SQLException;
    }

    private class ConnectionFactoryDecorator
    implements ConnectionFactory {
        private final ConnectionFactory defaultConnectionFactory;
        private ConnectionFactory customConnectionFactory;

        private ConnectionFactoryDecorator(ConnectionFactory connectionFactory) {
            this.defaultConnectionFactory = connectionFactory;
        }

        @Override
        public Connection connect(JdbcConfiguration config) throws SQLException {
            if (Strings.isNullOrEmpty(config.getConnectionFactoryClassName())) {
                return this.defaultConnectionFactory.connect(config);
            }
            if (this.customConnectionFactory == null) {
                this.customConnectionFactory = config.getInstance(JdbcConfiguration.CONNECTION_FACTORY_CLASS, ConnectionFactory.class);
            }
            return this.customConnectionFactory.connect(config);
        }
    }

    @FunctionalInterface
    public static interface StatementFactory {
        public Statement createStatement(Connection var1) throws SQLException;
    }

    public static interface ResultSetConsumer {
        public void accept(ResultSet var1) throws SQLException;
    }

    @FunctionalInterface
    public static interface CallPreparer {
        public void accept(CallableStatement var1) throws SQLException;
    }

    public static interface StatementPreparer {
        public void accept(PreparedStatement var1) throws SQLException;
    }

    public static interface BlockingMultiResultSetConsumer {
        public void accept(ResultSet[] var1) throws SQLException, InterruptedException;
    }

    public static interface BlockingResultSetConsumer {
        public void accept(ResultSet var1) throws SQLException, InterruptedException;
    }

    public static interface ParameterResultSetConsumer {
        public void accept(List<?> var1, ResultSet var2) throws SQLException;
    }

    @FunctionalInterface
    public static interface ResultSetExtractor<T> {
        public T apply(ResultSet var1) throws SQLException;
    }

    public static interface MultiResultSetConsumer {
        public void accept(ResultSet[] var1) throws SQLException;
    }
}

