/*
 * Decompiled with CFR 0.152.
 */
package com.hazelcast.cp.internal.persistence;

import com.hazelcast.core.HazelcastException;
import com.hazelcast.cp.internal.datastructures.snapshot.ChunkUtil;
import com.hazelcast.cp.internal.persistence.BufferedRaf;
import com.hazelcast.cp.internal.persistence.FileIOSupport;
import com.hazelcast.cp.internal.persistence.LogEntryRingBuffer;
import com.hazelcast.cp.internal.raft.impl.RaftEndpoint;
import com.hazelcast.cp.internal.raft.impl.log.LogEntry;
import com.hazelcast.cp.internal.raft.impl.log.SnapshotChunk;
import com.hazelcast.cp.internal.raft.impl.log.SnapshotEntry;
import com.hazelcast.cp.internal.raft.impl.persistence.LogFileStructure;
import com.hazelcast.cp.internal.raft.impl.persistence.RaftStateStore;
import com.hazelcast.internal.nio.IOUtil;
import com.hazelcast.internal.serialization.InternalSerializationService;
import com.hazelcast.internal.serialization.impl.SerializationUtil;
import java.io.File;
import java.io.IOException;
import java.io.RandomAccessFile;
import java.util.Collection;
import javax.annotation.Nonnull;
import javax.annotation.Nullable;

public class OnDiskRaftStateStore
implements RaftStateStore {
    private final File baseDir;
    private final InternalSerializationService serializationService;
    private final LogEntryRingBuffer logEntryRingBuffer;
    private BufferedRaf logRaf;
    private BufferedRaf.BufRafObjectDataOut logDataOut;
    private File currentFile;
    private File chunkFile;
    private BufferedRaf chunkRaf;
    private BufferedRaf.BufRafObjectDataOut chunkDataOut;
    private long nextEntryIndex;

    public OnDiskRaftStateStore(@Nonnull File baseDir, @Nonnull InternalSerializationService serializationService, int maxUncommittedEntries, @Nullable LogFileStructure logFileStructure) {
        this.baseDir = baseDir;
        this.serializationService = serializationService;
        this.logEntryRingBuffer = this.initializeRingBuffer(logFileStructure, maxUncommittedEntries);
        this.currentFile = logFileStructure != null ? new File(baseDir, logFileStructure.filename()) : null;
        this.nextEntryIndex = this.calculateNextEntryIndex(logFileStructure);
    }

    private LogEntryRingBuffer initializeRingBuffer(@Nullable LogFileStructure logFileStructure, int maxUncommittedEntries) {
        return logFileStructure != null ? new LogEntryRingBuffer(maxUncommittedEntries, logFileStructure) : new LogEntryRingBuffer(maxUncommittedEntries);
    }

    private long calculateNextEntryIndex(@Nullable LogFileStructure logFileStructure) {
        if (logFileStructure == null) {
            return 1L;
        }
        long[] tailEntryOffsets = logFileStructure.tailEntryOffsets();
        return logFileStructure.indexOfFirstTailEntry() + (long)tailEntryOffsets.length;
    }

    @Override
    public void open() throws IOException {
        if (!(this.baseDir.exists() || this.baseDir.mkdir() || this.baseDir.exists())) {
            throw new IOException("Cannot create directory: " + this.baseDir.getAbsolutePath());
        }
        if (this.currentFile == null) {
            this.currentFile = this.createFileWithIndex(this.nextEntryIndex);
        }
        this.logRaf = OnDiskRaftStateStore.openForAppend(this.currentFile);
        this.logDataOut = this.createObjectDataOutput(this.logRaf);
    }

    @Override
    public void persistEntry(@Nonnull LogEntry entry) throws IOException {
        this.validateEntryIndex(entry.index());
        this.logEntryRingBuffer.addEntryOffset(this.logRaf.filePointer());
        byte[] bytes = this.serializationService.toBytes(entry, 0, false);
        this.writeEntry(this.logRaf, this.logDataOut, bytes);
        ++this.nextEntryIndex;
    }

    @Override
    public void persistSnapshotChunk(Object snapshotChunk) throws IOException {
        SnapshotChunk chunk = (SnapshotChunk)snapshotChunk;
        this.validateChunkFile(chunk.index());
        this.initializeChunkFileIfNeeded(chunk.index());
        byte[] bytes = this.serializationService.toBytes(snapshotChunk, 0, false);
        this.writeEntry(this.chunkRaf, this.chunkDataOut, bytes);
    }

    @Override
    public void flushLogs() throws IOException {
        this.logDataOut.flush();
        this.logRaf.force();
        this.flushChunks();
        this.resetChunkState();
    }

    @Override
    public boolean isChunkingSupportedVersion() {
        return true;
    }

    @Override
    public void deleteSnapshotChunks(long snapshotIndex) {
        if (this.chunkFile == null || !OnDiskRaftStateStore.getRaftLogFileName(snapshotIndex).equals(this.chunkFile.getName())) {
            return;
        }
        IOUtil.delete(this.chunkFile);
        this.resetChunkState();
    }

    @Override
    public void persistSnapshot(@Nonnull SnapshotEntry snapshot) {
        throw new UnsupportedOperationException("Only chunked snapshot persistence is supported for versions > " + String.valueOf(ChunkUtil.MIN_CHUNKING_SUPPORTED_VERSION));
    }

    @Override
    public void deleteEntriesFrom(long startIndexInclusive) throws IOException {
        long rollbackOffset = this.logEntryRingBuffer.deleteEntriesFrom(startIndexInclusive);
        this.logRaf.seek(rollbackOffset);
        this.logRaf.setLength(rollbackOffset);
        this.nextEntryIndex = startIndexInclusive;
    }

    @Override
    public void persistInitialMembers(@Nonnull RaftEndpoint localMember, @Nonnull Collection<RaftEndpoint> initialMembers) throws IOException {
        this.writeToFile("members", out -> {
            out.writeObject(localMember);
            SerializationUtil.writeCollection(initialMembers, out);
        });
    }

    @Override
    public void persistTerm(int term, @Nullable RaftEndpoint votedFor) throws IOException {
        this.writeToFile("term", out -> {
            out.writeInt(term);
            out.writeObject(votedFor);
        });
    }

    @Override
    public void close() throws IOException {
        this.flushLogs();
        this.logRaf.close();
    }

    private void validateEntryIndex(long entryIndex) {
        if (entryIndex != this.nextEntryIndex) {
            throw new IllegalArgumentException(String.format("Expected entry index %,d, but got %,d", this.nextEntryIndex, entryIndex));
        }
    }

    private void validateChunkFile(long chunkIndex) {
        if (this.chunkFile != null && !OnDiskRaftStateStore.getRaftLogFileName(chunkIndex).equals(this.chunkFile.getName())) {
            throw new HazelcastException(String.format("Unexpected chunk file name: expected=%s, found=%s", OnDiskRaftStateStore.getRaftLogFileName(chunkIndex), this.chunkFile.getName()));
        }
    }

    private void initializeChunkFileIfNeeded(long chunkIndex) throws IOException {
        if (this.chunkFile == null) {
            this.chunkFile = this.createFileWithIndex(chunkIndex);
            this.chunkRaf = OnDiskRaftStateStore.openForAppend(this.chunkFile);
            this.chunkDataOut = this.createObjectDataOutput(this.chunkRaf);
        }
    }

    private void flushChunks() throws IOException {
        if (this.chunkFile == null) {
            return;
        }
        long snapshotIndex = OnDiskRaftStateStore.toEntryIndexFromFileName(this.chunkFile.getName());
        long newStartOffset = this.chunkRaf.filePointer();
        if (this.logEntryRingBuffer.topIndex() > snapshotIndex) {
            long copyFromOffset = this.logEntryRingBuffer.getEntryOffset(snapshotIndex + 1L);
            this.logRaf.seek(copyFromOffset);
            this.logRaf.copyTo(this.chunkRaf);
        }
        this.logRaf.close();
        this.logRaf = this.chunkRaf;
        this.logRaf.flush();
        this.logDataOut = this.chunkDataOut;
        this.logEntryRingBuffer.adjustToNewFile(newStartOffset, snapshotIndex);
        this.nextEntryIndex = Math.max(this.nextEntryIndex, snapshotIndex + 1L);
        IOUtil.delete(this.currentFile);
        this.currentFile = this.chunkFile;
    }

    private void resetChunkState() {
        this.chunkFile = null;
        this.chunkRaf = null;
        this.chunkDataOut = null;
    }

    private void writeEntry(BufferedRaf raf, BufferedRaf.BufRafObjectDataOut dataOut, byte[] bytes) throws IOException {
        raf.writeInt(bytes.length);
        dataOut.write(bytes);
        dataOut.writeCrc32();
    }

    private void writeToFile(String filename, FileIOSupport.Writable writable) throws IOException {
        FileIOSupport.writeWithChecksum(this.baseDir, filename, this.serializationService, writable);
    }

    private BufferedRaf.BufRafObjectDataOut createObjectDataOutput(BufferedRaf raf) {
        return raf.asObjectDataOutputStream(this.serializationService);
    }

    private File createFileWithIndex(long entryIndex) {
        File newFile = new File(this.baseDir, OnDiskRaftStateStore.getRaftLogFileName(entryIndex));
        if (this.currentFile != null && this.currentFile.getName().equals(newFile.getName())) {
            throw new IllegalArgumentException("Invalid index for new file: " + entryIndex);
        }
        return newFile;
    }

    private static BufferedRaf openForAppend(File file) throws IOException {
        BufferedRaf raf = new BufferedRaf(new RandomAccessFile(file, "rw"));
        raf.seek(raf.length());
        return raf;
    }

    static String getRaftLogFileName(long entryIndex) {
        return String.format("raftlog-%016x", entryIndex);
    }

    public static long toEntryIndexFromFileName(String fileName) {
        if (fileName == null || !fileName.startsWith("raftlog-")) {
            throw new IllegalArgumentException("Invalid raft log file name: " + fileName);
        }
        String hexPart = fileName.substring("raftlog-".length());
        try {
            return Long.parseUnsignedLong(hexPart, 16);
        }
        catch (NumberFormatException e) {
            throw new IllegalArgumentException("Invalid format in file name: " + fileName, e);
        }
    }

    public void setCurrentFile(File currentFile) {
        this.currentFile = currentFile;
    }

    public void setNextEntryIndex(long nextEntryIndex) {
        this.nextEntryIndex = nextEntryIndex;
    }

    public void setLogDataOut(BufferedRaf.BufRafObjectDataOut logDataOut) {
        this.logDataOut = logDataOut;
    }

    public void setLogRaf(BufferedRaf logRaf) {
        this.logRaf = logRaf;
    }
}

