/*
 * Decompiled with CFR 0.152.
 */
package com.hazelcast.cp;

import com.hazelcast.cp.internal.datastructures.snapshot.ChunkUtil;
import com.hazelcast.cp.internal.persistence.CPMetadataStoreImpl;
import com.hazelcast.cp.internal.persistence.OnDiskRaftStateLoader;
import com.hazelcast.cp.internal.persistence.VersionAwareOnDiskRaftStateStore;
import com.hazelcast.cp.internal.raft.impl.log.LogEntry;
import com.hazelcast.cp.internal.raft.impl.log.SnapshotEntry;
import com.hazelcast.cp.internal.raft.impl.persistence.RestoredRaftState;
import com.hazelcast.internal.cluster.Versions;
import com.hazelcast.internal.serialization.InternalSerializationService;
import com.hazelcast.internal.serialization.impl.DefaultSerializationServiceBuilder;
import com.hazelcast.logging.ILogger;
import com.hazelcast.logging.Logger;
import com.hazelcast.version.Version;
import java.io.IOException;
import java.io.UncheckedIOException;
import java.nio.file.FileVisitOption;
import java.nio.file.FileVisitResult;
import java.nio.file.FileVisitor;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.SimpleFileVisitor;
import java.nio.file.StandardCopyOption;
import java.nio.file.attribute.BasicFileAttributes;
import java.nio.file.attribute.FileAttribute;
import java.util.ArrayList;
import java.util.Comparator;
import java.util.List;
import java.util.Objects;
import java.util.stream.Stream;

public final class RaftLogFileExporter {
    private static final ILogger LOGGER = Logger.getLogger(RaftLogFileExporter.class);
    private static final String VERSION = "1.0";
    private static final Version DEFAULT_SOURCE_VERSION = Versions.V5_6;
    private static final Version DEFAULT_TARGET_VERSION = Versions.V5_5;
    private final Version sourceClusterVersion;
    private final Version targetClusterVersion;
    private final int maxUncommittedEntries;
    private final InternalSerializationService serializationService;

    public RaftLogFileExporter(Version sourceClusterVersion, Version targetClusterVersion, int maxUncommittedEntries) {
        this.sourceClusterVersion = Objects.requireNonNull(sourceClusterVersion);
        this.targetClusterVersion = Objects.requireNonNull(targetClusterVersion);
        if (maxUncommittedEntries <= 0) {
            throw new IllegalArgumentException("Max uncommitted entries must be positive");
        }
        this.maxUncommittedEntries = maxUncommittedEntries;
        this.serializationService = new DefaultSerializationServiceBuilder().build();
    }

    public static void main(String[] args) {
        if (args.length == 1 && ("--help".equals(args[0]) || "-h".equals(args[0]))) {
            RaftLogFileExporter.printUsage();
            System.exit(0);
        }
        try {
            CommandLineArguments cliArgs = RaftLogFileExporter.parseArguments(args);
            Path baseDir = Path.of(cliArgs.sourceDirectory, new String[0]);
            RaftLogFileExporter.validateInputs(baseDir, cliArgs.outputDirectory);
            RaftLogFileExporter exporter = new RaftLogFileExporter(Version.of(cliArgs.sourceVersion), Version.of(cliArgs.targetVersion), cliArgs.maxUncommittedEntries);
            exporter.processDirectory(baseDir, cliArgs.outputDirectory);
            System.out.println("\n\u2705 Export completed successfully!");
        }
        catch (IllegalArgumentException e) {
            System.err.println("Error: " + e.getMessage());
            RaftLogFileExporter.printUsage();
            System.exit(1);
        }
        catch (Exception e) {
            System.err.println("\n\u274c Export failed: " + e.getMessage());
            LOGGER.severe("Export error", e);
            System.exit(1);
        }
    }

    private static void printUsage() {
        System.out.printf("RaftLogFileExporter v%s - Hazelcast Raft Log Migration Tool%n%n", VERSION);
        System.out.println("Usage:");
        System.out.println("  java RaftLogFileExporter --source=<dir> --output=<dir> --max-uncommitted-entries=<n> [options]");
        System.out.println("\nRequired arguments:");
        System.out.println("  --source=<dir>                Root directory containing Raft logs");
        System.out.println("  --output=<dir>                Output directory name (must not exist)");
        System.out.println("  --max-uncommitted-entries=<n> Configured max uncommitted entries (must be positive)");
        System.out.println("\nOptions:");
        System.out.println("  --sourceVersion=<ver>        Source Hazelcast version (must be: " + String.valueOf(DEFAULT_SOURCE_VERSION) + ")");
        System.out.println("  --targetVersion=<ver>        Target Hazelcast version (must be: " + String.valueOf(DEFAULT_TARGET_VERSION) + ")");
        System.out.println("  -h, --help                   Show this help");
        System.out.println("\nExample:");
        System.out.println("  java RaftLogFileExporter --source=/data/hz --output=migrated \\");
        System.out.println("                           --max-uncommitted-entries=10000");
    }

    private static CommandLineArguments parseArguments(String[] args) {
        CommandLineArguments cliArgs = new CommandLineArguments();
        boolean maxEntriesProvided = false;
        for (String arg : args) {
            if (arg.startsWith("--source=")) {
                cliArgs.sourceDirectory = arg.substring("--source=".length());
                continue;
            }
            if (arg.startsWith("--output=")) {
                cliArgs.outputDirectory = arg.substring("--output=".length());
                continue;
            }
            if (arg.startsWith("--sourceVersion=")) {
                cliArgs.sourceVersion = arg.substring("--sourceVersion=".length());
                continue;
            }
            if (arg.startsWith("--targetVersion=")) {
                cliArgs.targetVersion = arg.substring("--targetVersion=".length());
                continue;
            }
            if (arg.startsWith("--max-uncommitted-entries=")) {
                cliArgs.maxUncommittedEntries = Integer.parseInt(arg.substring("--max-uncommitted-entries=".length()));
                maxEntriesProvided = true;
                continue;
            }
            if (arg.equals("-h") || arg.equals("--help")) {
                RaftLogFileExporter.printUsage();
                System.exit(0);
                continue;
            }
            throw new IllegalArgumentException("Unknown argument: " + arg);
        }
        if (cliArgs.sourceDirectory == null || cliArgs.outputDirectory == null || !maxEntriesProvided) {
            throw new IllegalArgumentException("--source, --output, and --max-uncommitted-entries are all required arguments");
        }
        if (!DEFAULT_SOURCE_VERSION.toString().equals(cliArgs.sourceVersion)) {
            throw new IllegalArgumentException("Only source version " + String.valueOf(DEFAULT_SOURCE_VERSION) + " is supported");
        }
        if (!DEFAULT_TARGET_VERSION.toString().equals(cliArgs.targetVersion)) {
            throw new IllegalArgumentException("Only target version " + String.valueOf(DEFAULT_TARGET_VERSION) + " is supported");
        }
        return cliArgs;
    }

    private static void validateInputs(Path sourceDir, String outputDirName) {
        if (outputDirName == null || outputDirName.trim().isEmpty()) {
            throw new IllegalArgumentException("Output directory name cannot be empty");
        }
        if (outputDirName.contains("/") || outputDirName.contains("\\") || outputDirName.contains("..")) {
            throw new IllegalArgumentException("Output directory name cannot contain path separators");
        }
        if (!Files.isDirectory(sourceDir, new LinkOption[0])) {
            throw new IllegalArgumentException("Source path is not a directory: " + String.valueOf(sourceDir));
        }
        Path outputDir = sourceDir.getParent().resolve(outputDirName);
        if (Files.exists(outputDir, new LinkOption[0])) {
            throw new IllegalArgumentException("Output directory already exists: " + String.valueOf(outputDir));
        }
    }

    public void processDirectory(Path sourceDir, String outputDirName) throws IOException {
        System.out.println("\ud83d\udd0e Starting Raft log export process...");
        System.out.println("Source directory: " + String.valueOf(sourceDir));
        System.out.println("Source version: " + String.valueOf(this.sourceClusterVersion));
        System.out.println("Target version: " + String.valueOf(this.targetClusterVersion));
        System.out.println("Output directory name: " + outputDirName);
        Path outputDir = sourceDir.getParent().resolve(outputDirName);
        this.prepareOutputDirectory(outputDir);
        List<Path> cpDirs = RaftLogFileExporter.findCpDirectories(sourceDir);
        if (cpDirs.isEmpty()) {
            System.out.println("No CP member directories found to process. Exiting.");
            return;
        }
        System.out.println("Found " + cpDirs.size() + " CP member directories.");
        for (Path cpDir : cpDirs) {
            Path relativePath = sourceDir.relativize(cpDir);
            Path outputCpDir = outputDir.resolve(relativePath);
            System.out.println("\nCopying structure for: " + String.valueOf(cpDir.getFileName()));
            RaftLogFileExporter.copyDirectoryStructure(cpDir, outputCpDir);
        }
        List<Path> raftLogFiles = RaftLogFileExporter.findRaftLogFiles(sourceDir);
        System.out.println("\nFound " + raftLogFiles.size() + " original Raft log files to export.");
        for (Path sourceLogFile : raftLogFiles) {
            Path relativePath = sourceDir.relativize(sourceLogFile.getParent());
            Path targetLogDir = outputDir.resolve(relativePath);
            this.exportSingleFile(sourceLogFile, targetLogDir);
        }
    }

    public void exportSingleFile(Path sourceFile, Path targetLogDir) throws IOException {
        System.out.println("   -> Exporting " + String.valueOf(sourceFile.getFileName()) + "...");
        this.validateSourceFile(sourceFile);
        Files.createDirectories(targetLogDir, new FileAttribute[0]);
        OnDiskRaftStateLoader loader = new OnDiskRaftStateLoader(sourceFile.getParent().toFile(), this.maxUncommittedEntries, this.serializationService, Logger.getLogger(OnDiskRaftStateLoader.class));
        RestoredRaftState restoredState = loader.load();
        try (VersionAwareOnDiskRaftStateStore targetStore = new VersionAwareOnDiskRaftStateStore(targetLogDir.toFile(), this.serializationService, this.maxUncommittedEntries, null, () -> this.targetClusterVersion);){
            targetStore.open();
            this.exportRaftState(restoredState, targetStore);
        }
        LOGGER.info("Exported Raft log: " + String.valueOf(sourceFile.getFileName()));
    }

    private void exportRaftState(RestoredRaftState sourceState, VersionAwareOnDiskRaftStateStore targetStore) throws IOException {
        targetStore.persistTerm(sourceState.term(), sourceState.votedFor());
        targetStore.persistInitialMembers(sourceState.localEndpoint(), sourceState.initialMembers());
        SnapshotEntry snapshotEntry = sourceState.snapshot();
        if (snapshotEntry != null) {
            targetStore.persistSnapshot(ChunkUtil.toV55SnapshotFromChunkedV60(snapshotEntry));
        }
        for (LogEntry entry : sourceState.entries()) {
            targetStore.persistEntry(entry);
        }
    }

    private void prepareOutputDirectory(Path outputDir) throws IOException {
        if (Files.exists(outputDir, new LinkOption[0])) {
            System.out.println("Deleting existing output directory: " + String.valueOf(outputDir));
            RaftLogFileExporter.deleteRecursively(outputDir);
        }
        System.out.println("Creating clean output directory: " + String.valueOf(outputDir));
        Files.createDirectories(outputDir, new FileAttribute[0]);
    }

    private void validateSourceFile(Path sourceFile) throws IOException {
        if (!Files.exists(sourceFile, new LinkOption[0])) {
            throw new IOException("Source file does not exist: " + String.valueOf(sourceFile));
        }
        if (!Files.isRegularFile(sourceFile, new LinkOption[0])) {
            throw new IOException("Source path is not a file: " + String.valueOf(sourceFile));
        }
        if (!Files.isReadable(sourceFile)) {
            throw new IOException("Source file is not readable: " + String.valueOf(sourceFile));
        }
    }

    private static void deleteRecursively(Path path) throws IOException {
        try (Stream<Path> walk = Files.walk(path, new FileVisitOption[0]);){
            walk.sorted(Comparator.reverseOrder()).forEach(p -> {
                try {
                    Files.delete(p);
                }
                catch (IOException e) {
                    throw new UncheckedIOException("Failed to delete " + String.valueOf(p), e);
                }
            });
        }
    }

    private static void copyDirectoryStructure(Path source, Path target) throws IOException {
        Files.walkFileTree(source, new DirectoryStructureCopier(source, target));
    }

    private static List<Path> findCpDirectories(Path baseDir) throws IOException {
        ArrayList<Path> cpDirs = new ArrayList<Path>();
        try (Stream<Path> stream = Files.list(baseDir);){
            stream.filter(x$0 -> Files.isDirectory(x$0, new LinkOption[0])).filter(dir -> CPMetadataStoreImpl.isCPDirectory(dir.toFile())).forEach(cpDirs::add);
        }
        return cpDirs;
    }

    private static List<Path> findRaftLogFiles(Path baseDir) throws IOException {
        final ArrayList<Path> raftLogFiles = new ArrayList<Path>();
        Files.walkFileTree(baseDir, (FileVisitor<? super Path>)new SimpleFileVisitor<Path>(){

            @Override
            public FileVisitResult visitFile(Path file, BasicFileAttributes attrs) {
                if (file.getFileName().toString().startsWith("raftlog-")) {
                    raftLogFiles.add(file);
                }
                return FileVisitResult.CONTINUE;
            }
        });
        return raftLogFiles;
    }

    private static class CommandLineArguments {
        String sourceDirectory;
        String outputDirectory;
        String sourceVersion = DEFAULT_SOURCE_VERSION.toString();
        String targetVersion = DEFAULT_TARGET_VERSION.toString();
        int maxUncommittedEntries;

        private CommandLineArguments() {
        }
    }

    private static class DirectoryStructureCopier
    extends SimpleFileVisitor<Path> {
        private final Path source;
        private final Path target;

        DirectoryStructureCopier(Path source, Path target) {
            this.source = source;
            this.target = target;
        }

        @Override
        public FileVisitResult preVisitDirectory(Path dir, BasicFileAttributes attrs) throws IOException {
            Path targetDir = this.target.resolve(this.source.relativize(dir));
            try {
                Files.createDirectories(targetDir, new FileAttribute[0]);
            }
            catch (IOException e) {
                throw new IOException("Failed to create directory: " + String.valueOf(targetDir), e);
            }
            return FileVisitResult.CONTINUE;
        }

        @Override
        public FileVisitResult visitFile(Path file, BasicFileAttributes attrs) throws IOException {
            String fileName = file.getFileName().toString();
            if (fileName.startsWith("raftlog-") || fileName.equals("members") || fileName.equals("term")) {
                LOGGER.fine("Skipping Raft state file: " + fileName);
                return FileVisitResult.CONTINUE;
            }
            Path targetFile = this.target.resolve(this.source.relativize(file));
            try {
                Files.copy(file, targetFile, StandardCopyOption.REPLACE_EXISTING);
            }
            catch (IOException e) {
                throw new IOException("Failed to copy file: " + String.valueOf(file) + " to " + String.valueOf(targetFile), e);
            }
            return FileVisitResult.CONTINUE;
        }
    }
}

