/*
 * Decompiled with CFR 0.152.
 */
package org.apache.kafka.clients.admin.internals;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.OptionalInt;
import java.util.Set;
import java.util.function.BiFunction;
import java.util.function.Function;
import java.util.stream.Collectors;
import org.apache.kafka.clients.admin.internals.AdminApiFuture;
import org.apache.kafka.clients.admin.internals.AdminApiHandler;
import org.apache.kafka.clients.admin.internals.AdminApiLookupStrategy;
import org.apache.kafka.clients.admin.internals.ApiRequestScope;
import org.apache.kafka.clients.admin.internals.CoordinatorStrategy;
import org.apache.kafka.common.Node;
import org.apache.kafka.common.errors.DisconnectException;
import org.apache.kafka.common.errors.UnsupportedVersionException;
import org.apache.kafka.common.requests.AbstractRequest;
import org.apache.kafka.common.requests.AbstractResponse;
import org.apache.kafka.common.requests.FindCoordinatorRequest;
import org.apache.kafka.common.requests.OffsetFetchRequest;
import org.apache.kafka.common.utils.ExponentialBackoff;
import org.apache.kafka.common.utils.LogContext;
import org.slf4j.Logger;

public class AdminApiDriver<K, V> {
    private final Logger log;
    private final ExponentialBackoff retryBackoff;
    private final long deadlineMs;
    private final AdminApiHandler<K, V> handler;
    private final AdminApiFuture<K, V> future;
    private final BiMultimap<ApiRequestScope, K> lookupMap = new BiMultimap();
    private final BiMultimap<FulfillmentScope, K> fulfillmentMap = new BiMultimap();
    private final Map<ApiRequestScope, RequestState> requestStates = new HashMap<ApiRequestScope, RequestState>();

    public AdminApiDriver(AdminApiHandler<K, V> handler, AdminApiFuture<K, V> future, long deadlineMs, long retryBackoffMs, long retryBackoffMaxMs, LogContext logContext) {
        this.handler = handler;
        this.future = future;
        this.deadlineMs = deadlineMs;
        this.retryBackoff = new ExponentialBackoff(retryBackoffMs, 2, retryBackoffMaxMs, 0.2);
        this.log = logContext.logger(AdminApiDriver.class);
        this.retryLookup(future.lookupKeys());
    }

    private void map(K key, Integer brokerId) {
        this.lookupMap.remove(key);
        this.fulfillmentMap.put(new FulfillmentScope(brokerId), (FulfillmentScope)key);
    }

    private void unmap(K key) {
        this.fulfillmentMap.remove(key);
        ApiRequestScope lookupScope = this.handler.lookupStrategy().lookupScope(key);
        OptionalInt destinationBrokerId = lookupScope.destinationBrokerId();
        if (destinationBrokerId.isPresent()) {
            this.fulfillmentMap.put(new FulfillmentScope(destinationBrokerId.getAsInt()), (FulfillmentScope)key);
        } else {
            this.lookupMap.put(this.handler.lookupStrategy().lookupScope(key), (ApiRequestScope)key);
        }
    }

    private void clear(Collection<K> keys) {
        keys.forEach(key -> {
            this.lookupMap.remove(key);
            this.fulfillmentMap.remove(key);
        });
    }

    OptionalInt keyToBrokerId(K key) {
        Optional<FulfillmentScope> scope = this.fulfillmentMap.getKey(key);
        return scope.map(fulfillmentScope -> OptionalInt.of(fulfillmentScope.destinationBrokerId)).orElseGet(OptionalInt::empty);
    }

    private void completeExceptionally(Map<K, Throwable> errors) {
        if (!errors.isEmpty()) {
            this.future.completeExceptionally(errors);
            this.clear(errors.keySet());
        }
    }

    private void completeLookupExceptionally(Map<K, Throwable> errors) {
        if (!errors.isEmpty()) {
            this.future.completeLookupExceptionally(errors);
            this.clear(errors.keySet());
        }
    }

    private void retryLookup(Collection<K> keys) {
        keys.forEach(this::unmap);
    }

    private void complete(Map<K, V> values) {
        if (!values.isEmpty()) {
            this.future.complete(values);
            this.clear(values.keySet());
        }
    }

    private void completeLookup(Map<K, Integer> brokerIdMapping) {
        if (!brokerIdMapping.isEmpty()) {
            this.future.completeLookup(brokerIdMapping);
            brokerIdMapping.forEach(this::map);
        }
    }

    public List<RequestSpec<K>> poll() {
        ArrayList<RequestSpec<K>> requests = new ArrayList<RequestSpec<K>>();
        this.collectLookupRequests(requests);
        this.collectFulfillmentRequests(requests);
        return requests;
    }

    public void onResponse(long currentTimeMs, RequestSpec<K> spec, AbstractResponse response, Node node) {
        this.clearInflightRequest(currentTimeMs, spec);
        if (spec.scope instanceof FulfillmentScope) {
            AdminApiHandler.ApiResult<K, V> result = this.handler.handleResponse(node, spec.keys, response);
            this.complete(result.completedKeys);
            this.completeExceptionally(result.failedKeys);
            this.retryLookup(result.unmappedKeys);
        } else {
            AdminApiLookupStrategy.LookupResult<K> result = this.handler.lookupStrategy().handleResponse(spec.keys, response);
            result.completedKeys.forEach(this.lookupMap::remove);
            this.completeLookup(result.mappedKeys);
            this.completeLookupExceptionally(result.failedKeys);
        }
    }

    public void onFailure(long currentTimeMs, RequestSpec<K> spec, Throwable t2) {
        this.clearInflightRequest(currentTimeMs, spec);
        if (t2 instanceof DisconnectException) {
            this.log.debug("Node disconnected before response could be received for request {}. Will attempt retry", (Object)spec.request);
            Set keysToUnmap = spec.keys.stream().filter(this.future.lookupKeys()::contains).collect(Collectors.toSet());
            this.retryLookup(keysToUnmap);
        } else if (t2 instanceof FindCoordinatorRequest.NoBatchedFindCoordinatorsException || t2 instanceof OffsetFetchRequest.NoBatchedOffsetFetchRequestException) {
            ((CoordinatorStrategy)this.handler.lookupStrategy()).disableBatch();
            Set keysToUnmap = spec.keys.stream().filter(this.future.lookupKeys()::contains).collect(Collectors.toSet());
            this.retryLookup(keysToUnmap);
        } else if (t2 instanceof UnsupportedVersionException) {
            if (spec.scope instanceof FulfillmentScope) {
                int brokerId = ((FulfillmentScope)spec.scope).destinationBrokerId;
                Map<K, Throwable> unrecoverableFailures = this.handler.handleUnsupportedVersionException(brokerId, (UnsupportedVersionException)t2, spec.keys);
                this.completeExceptionally(unrecoverableFailures);
            } else {
                Map unrecoverableLookupFailures = this.handler.lookupStrategy().handleUnsupportedVersionException((UnsupportedVersionException)t2, spec.keys);
                this.completeLookupExceptionally(unrecoverableLookupFailures);
                Set keysToUnmap = spec.keys.stream().filter(k -> !unrecoverableLookupFailures.containsKey(k)).collect(Collectors.toSet());
                this.retryLookup(keysToUnmap);
            }
        } else {
            Map errors = spec.keys.stream().collect(Collectors.toMap(Function.identity(), key -> t2));
            if (spec.scope instanceof FulfillmentScope) {
                this.completeExceptionally(errors);
            } else {
                this.completeLookupExceptionally(errors);
            }
        }
    }

    private void clearInflightRequest(long currentTimeMs, RequestSpec<K> spec) {
        RequestState requestState = this.requestStates.get(spec.scope);
        if (requestState != null) {
            if (spec.scope instanceof FulfillmentScope) {
                requestState.clearInflightAndBackoff(currentTimeMs);
            } else {
                requestState.clearInflight(currentTimeMs);
            }
        }
    }

    private <T extends ApiRequestScope> void collectRequests(List<RequestSpec<K>> requests, BiMultimap<T, K> multimap, BiFunction<Set<K>, T, Collection<AdminApiHandler.RequestAndKeys<K>>> buildRequest) {
        for (Map.Entry<T, Set<K>> entry : multimap.entrySet()) {
            RequestState requestState;
            ApiRequestScope scope = (ApiRequestScope)entry.getKey();
            Set<K> keys = entry.getValue();
            if (keys.isEmpty() || (requestState = this.requestStates.computeIfAbsent(scope, c -> new RequestState())).hasInflight()) continue;
            Set<K> copyKeys = Collections.unmodifiableSet(new HashSet<K>(keys));
            Collection<AdminApiHandler.RequestAndKeys<K>> newRequests = buildRequest.apply(copyKeys, (Set<K>)((Object)scope));
            if (newRequests.isEmpty()) {
                return;
            }
            AdminApiHandler.RequestAndKeys<K> newRequest = newRequests.iterator().next();
            RequestSpec spec = new RequestSpec(this.handler.apiName() + "(api=" + (Object)((Object)newRequest.request.apiKey()) + ")", scope, newRequest.keys, newRequest.request, requestState.nextAllowedRetryMs, this.deadlineMs, requestState.tries);
            requestState.setInflight(spec);
            requests.add(spec);
        }
    }

    private void collectLookupRequests(List<RequestSpec<K>> requests) {
        this.collectRequests(requests, this.lookupMap, (keys, scope) -> Collections.singletonList(new AdminApiHandler.RequestAndKeys(this.handler.lookupStrategy().buildRequest((Set<K>)keys), keys)));
    }

    private void collectFulfillmentRequests(List<RequestSpec<K>> requests) {
        this.collectRequests(requests, this.fulfillmentMap, (keys, scope) -> this.handler.buildRequest(scope.destinationBrokerId, (Set<K>)keys));
    }

    private static class BiMultimap<K, V> {
        private final Map<V, K> reverseMap = new HashMap<V, K>();
        private final Map<K, Set<V>> map = new HashMap<K, Set<V>>();

        private BiMultimap() {
        }

        void put(K key, V value) {
            this.remove(value);
            this.reverseMap.put(value, key);
            this.map.computeIfAbsent(key, k -> new HashSet()).add(value);
        }

        void remove(V value) {
            Set<V> set;
            K key = this.reverseMap.remove(value);
            if (key != null && (set = this.map.get(key)) != null) {
                set.remove(value);
                if (set.isEmpty()) {
                    this.map.remove(key);
                }
            }
        }

        Optional<K> getKey(V value) {
            return Optional.ofNullable(this.reverseMap.get(value));
        }

        Set<Map.Entry<K, Set<V>>> entrySet() {
            return this.map.entrySet();
        }
    }

    private static class FulfillmentScope
    implements ApiRequestScope {
        public final int destinationBrokerId;

        private FulfillmentScope(int destinationBrokerId) {
            this.destinationBrokerId = destinationBrokerId;
        }

        @Override
        public OptionalInt destinationBrokerId() {
            return OptionalInt.of(this.destinationBrokerId);
        }

        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || this.getClass() != o.getClass()) {
                return false;
            }
            FulfillmentScope that = (FulfillmentScope)o;
            return this.destinationBrokerId == that.destinationBrokerId;
        }

        public int hashCode() {
            return Objects.hash(this.destinationBrokerId);
        }
    }

    private class RequestState {
        private Optional<RequestSpec<K>> inflightRequest = Optional.empty();
        private int tries = 0;
        private long nextAllowedRetryMs = 0L;

        private RequestState() {
        }

        boolean hasInflight() {
            return this.inflightRequest.isPresent();
        }

        public void clearInflight(long currentTimeMs) {
            this.inflightRequest = Optional.empty();
            this.nextAllowedRetryMs = currentTimeMs;
        }

        public void clearInflightAndBackoff(long currentTimeMs) {
            this.clearInflight(currentTimeMs + AdminApiDriver.this.retryBackoff.backoff(this.tries >= 1 ? (long)(this.tries - 1) : 0L));
        }

        public void setInflight(RequestSpec<K> spec) {
            this.inflightRequest = Optional.of(spec);
            ++this.tries;
        }
    }

    public static class RequestSpec<K> {
        public final String name;
        public final ApiRequestScope scope;
        public final Set<K> keys;
        public final AbstractRequest.Builder<?> request;
        public final long nextAllowedTryMs;
        public final long deadlineMs;
        public final int tries;

        public RequestSpec(String name, ApiRequestScope scope, Set<K> keys, AbstractRequest.Builder<?> request, long nextAllowedTryMs, long deadlineMs, int tries) {
            this.name = name;
            this.scope = scope;
            this.keys = keys;
            this.request = request;
            this.nextAllowedTryMs = nextAllowedTryMs;
            this.deadlineMs = deadlineMs;
            this.tries = tries;
        }

        public String toString() {
            return "RequestSpec(name=" + this.name + ", scope=" + this.scope + ", keys=" + this.keys + ", request=" + this.request + ", nextAllowedTryMs=" + this.nextAllowedTryMs + ", deadlineMs=" + this.deadlineMs + ", tries=" + this.tries + ')';
        }
    }
}

