/*
 * Decompiled with CFR 0.152.
 */
package com.hazelcast.spi.utils;

import com.hazelcast.internal.nio.IOUtil;
import com.hazelcast.spi.exception.RestClientException;
import com.hazelcast.spi.utils.RetryUtils;
import java.io.BufferedReader;
import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.net.URI;
import java.net.http.HttpClient;
import java.net.http.HttpRequest;
import java.net.http.HttpResponse;
import java.nio.charset.StandardCharsets;
import java.security.KeyStore;
import java.security.cert.Certificate;
import java.security.cert.CertificateException;
import java.security.cert.CertificateFactory;
import java.time.Duration;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Scanner;
import java.util.Set;
import javax.annotation.Nullable;
import javax.net.ssl.SSLContext;
import javax.net.ssl.TrustManagerFactory;

public final class RestClient {
    public static final int HTTP_OK = 200;
    public static final int HTTP_NOT_FOUND = 404;
    public static final int DEFAULT_CONNECT_TIMEOUT_SECONDS = -1;
    private static final String WATCH_FORMAT = "watch=1&resourceVersion=%s";
    private final String url;
    private final List<Parameter> headers = new ArrayList<Parameter>();
    private final HttpClient httpClient;
    private Set<Integer> expectedResponseCodes;
    private String body;
    private int requestTimeoutSeconds;
    private int retries;

    private RestClient(String url, @Nullable String caCertificate, int connectTimeoutSeconds) {
        this.url = url;
        HttpClient.Builder builder = HttpClient.newBuilder().version(HttpClient.Version.HTTP_1_1);
        if (connectTimeoutSeconds > 0) {
            builder.connectTimeout(Duration.ofSeconds(connectTimeoutSeconds));
        }
        if (caCertificate != null) {
            builder.sslContext(this.buildSslContext(caCertificate));
        }
        this.httpClient = builder.build();
    }

    public static RestClient create(String url) {
        return RestClient.create(url, -1);
    }

    public static RestClient create(String url, int connectTimeoutSeconds) {
        return new RestClient(url, null, connectTimeoutSeconds);
    }

    public static RestClient createWithSSL(String url, String caCertificate) {
        return RestClient.createWithSSL(url, caCertificate, -1);
    }

    public static RestClient createWithSSL(String url, String caCertificate, int connectTimeoutSeconds) {
        return new RestClient(url, caCertificate, connectTimeoutSeconds);
    }

    public RestClient withHeaders(Map<String, String> headers) {
        for (Map.Entry<String, String> entry : headers.entrySet()) {
            this.withHeader(entry.getKey(), entry.getValue());
        }
        return this;
    }

    public RestClient withHeader(String name, String value) {
        if (name.equalsIgnoreCase("host")) {
            return this;
        }
        this.headers.add(new Parameter(name, value));
        return this;
    }

    public RestClient withBody(String body) {
        this.body = body;
        return this;
    }

    public RestClient withRequestTimeoutSeconds(int timeoutSeconds) {
        this.requestTimeoutSeconds = timeoutSeconds;
        return this;
    }

    public RestClient withRetries(int retries) {
        this.retries = retries;
        return this;
    }

    public RestClient expectResponseCodes(Integer ... codes) {
        if (this.expectedResponseCodes == null) {
            this.expectedResponseCodes = new HashSet<Integer>();
        }
        this.expectedResponseCodes.addAll(Arrays.asList(codes));
        return this;
    }

    public Response get() {
        return this.callWithRetries(this.buildHttpRequest("GET"));
    }

    public Response post() {
        return this.callWithRetries(this.buildHttpRequest("POST"));
    }

    public Response put() {
        return this.callWithRetries(this.buildHttpRequest("PUT"));
    }

    private HttpRequest buildHttpRequest(String method) {
        HttpRequest.Builder builder = HttpRequest.newBuilder().uri(URI.create(this.url));
        if (this.requestTimeoutSeconds > 0) {
            builder.timeout(Duration.ofSeconds(this.requestTimeoutSeconds));
        }
        this.headers.forEach(parameter -> builder.header(parameter.key(), parameter.value()));
        HttpRequest.BodyPublisher publisher = this.body == null ? HttpRequest.BodyPublishers.noBody() : HttpRequest.BodyPublishers.ofString(this.body);
        String finalMethod = this.body != null && "GET".equals(method) ? "POST" : method;
        return builder.method(finalMethod, publisher).build();
    }

    private Response callWithRetries(HttpRequest request) {
        return RetryUtils.retry(() -> this.call(request), this.retries);
    }

    private Response call(HttpRequest request) {
        try {
            HttpResponse<String> response = this.httpClient.send(request, HttpResponse.BodyHandlers.ofString());
            this.checkResponseCode(request.method(), response);
            return new Response(response.statusCode(), response.body());
        }
        catch (IOException e) {
            throw new RestClientException("Failure in executing REST call", e);
        }
        catch (InterruptedException e) {
            throw new RestClientException("REST call interrupted", e);
        }
    }

    public WatchResponse watch(String resourceVersion) throws RestClientException {
        try {
            String appendWatchParameter = (this.url.contains("?") ? "&" : "?") + String.format(WATCH_FORMAT, resourceVersion);
            String completeUrl = this.url + appendWatchParameter;
            HttpRequest.Builder requestBuilder = HttpRequest.newBuilder().uri(URI.create(completeUrl)).GET();
            if (this.requestTimeoutSeconds > 0) {
                requestBuilder.timeout(Duration.ofSeconds(this.requestTimeoutSeconds));
            }
            for (Parameter header : this.headers) {
                requestBuilder.header(header.key(), header.value());
            }
            if (this.body != null) {
                byte[] bodyData = this.body.getBytes(StandardCharsets.UTF_8);
                requestBuilder.setHeader("Content-Length", String.valueOf(bodyData.length));
                requestBuilder.setHeader("charset", "utf-8");
                requestBuilder.method("GET", HttpRequest.BodyPublishers.ofByteArray(bodyData));
            }
            HttpRequest request = requestBuilder.build();
            HttpResponse<InputStream> response = this.httpClient.send(request, HttpResponse.BodyHandlers.ofInputStream());
            this.checkResponseCode(request.method(), response);
            return new WatchResponse(response);
        }
        catch (IOException e) {
            throw new RestClientException("Failure in executing REST call", e);
        }
        catch (InterruptedException e) {
            throw new RestClientException("REST call interrupted", e);
        }
    }

    private void checkResponseCode(String method, HttpResponse<?> response) {
        int responseCode = response.statusCode();
        if (!this.isExpectedResponseCode(responseCode)) {
            Object errorMessage = "none, body type: " + String.valueOf(response.body().getClass());
            if (response.body() instanceof String) {
                errorMessage = (String)response.body();
            } else {
                Object obj = response.body();
                if (obj instanceof InputStream) {
                    InputStream inputStream = (InputStream)obj;
                    Scanner scanner = new Scanner(inputStream, StandardCharsets.UTF_8);
                    scanner.useDelimiter("\\Z");
                    if (scanner.hasNext()) {
                        errorMessage = scanner.next();
                    }
                }
            }
            throw new RestClientException(String.format("Failure executing: %s at: %s. Message: %s", method, this.url, errorMessage), responseCode);
        }
    }

    private boolean isExpectedResponseCode(int responseCode) {
        return this.expectedResponseCodes == null ? responseCode == 200 : this.expectedResponseCodes.contains(responseCode);
    }

    private SSLContext buildSslContext(String caCertificate) {
        try {
            String keystoreType = System.getProperty("hazelcast.restclient.ca.storetype", "PKCS12");
            KeyStore keyStore = KeyStore.getInstance(keystoreType);
            keyStore.load(null, null);
            int i = 0;
            for (Certificate certificate : this.generateCertificates(caCertificate)) {
                String alias = String.format("ca-%d", i++);
                keyStore.setCertificateEntry(alias, certificate);
            }
            TrustManagerFactory tmf = TrustManagerFactory.getInstance(TrustManagerFactory.getDefaultAlgorithm());
            tmf.init(keyStore);
            SSLContext sSLContext = SSLContext.getInstance("TLS");
            sSLContext.init(null, tmf.getTrustManagers(), null);
            return sSLContext;
        }
        catch (Exception e) {
            throw new RestClientException("Failure in generating SSLSocketFactory for certificate " + caCertificate, e);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private Collection<? extends Certificate> generateCertificates(String caCertificate) throws CertificateException {
        Collection<? extends Certificate> collection;
        ByteArrayInputStream caInput = null;
        try {
            CertificateFactory cf = CertificateFactory.getInstance("X.509");
            caInput = new ByteArrayInputStream(caCertificate.getBytes(StandardCharsets.UTF_8));
            collection = cf.generateCertificates(caInput);
        }
        catch (Throwable throwable) {
            IOUtil.closeResource(caInput);
            throw throwable;
        }
        IOUtil.closeResource(caInput);
        return collection;
    }

    private record Parameter(String key, String value) {
    }

    public static class Response {
        private final int code;
        private final String body;

        Response(int code, String body) {
            this.code = code;
            this.body = body;
        }

        public int getCode() {
            return this.code;
        }

        public String getBody() {
            return this.body;
        }
    }

    public static class WatchResponse {
        private final int code;
        private final HttpResponse<InputStream> response;
        private final BufferedReader reader;

        public WatchResponse(HttpResponse<InputStream> response) {
            this.code = response.statusCode();
            this.response = response;
            this.reader = new BufferedReader(new InputStreamReader(response.body()));
        }

        public int getCode() {
            return this.code;
        }

        public String nextLine() throws IOException {
            return this.reader.readLine();
        }

        public void disconnect() throws IOException {
            this.response.body().close();
        }
    }
}

