/*
 * Decompiled with CFR 0.152.
 */
package software.amazon.awssdk.transfer.s3.internal;

import java.io.IOException;
import java.nio.file.FileSystem;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.attribute.FileAttribute;
import java.util.Collection;
import java.util.Queue;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CompletionException;
import java.util.concurrent.CompletionStage;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.function.Function;
import java.util.function.Predicate;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.services.s3.model.GetObjectRequest;
import software.amazon.awssdk.services.s3.model.ListObjectsV2Request;
import software.amazon.awssdk.services.s3.model.S3Object;
import software.amazon.awssdk.transfer.s3.S3TransferManager;
import software.amazon.awssdk.transfer.s3.internal.AsyncBufferingSubscriber;
import software.amazon.awssdk.transfer.s3.internal.DirectoryHelperUtils;
import software.amazon.awssdk.transfer.s3.internal.ListObjectsHelper;
import software.amazon.awssdk.transfer.s3.internal.TransferConfigurationOption;
import software.amazon.awssdk.transfer.s3.internal.TransferManagerConfiguration;
import software.amazon.awssdk.transfer.s3.internal.model.DefaultDirectoryDownload;
import software.amazon.awssdk.transfer.s3.model.CompletedDirectoryDownload;
import software.amazon.awssdk.transfer.s3.model.CompletedFileDownload;
import software.amazon.awssdk.transfer.s3.model.DirectoryDownload;
import software.amazon.awssdk.transfer.s3.model.DownloadDirectoryRequest;
import software.amazon.awssdk.transfer.s3.model.DownloadFileRequest;
import software.amazon.awssdk.transfer.s3.model.FailedFileDownload;
import software.amazon.awssdk.transfer.s3.model.FileDownload;
import software.amazon.awssdk.utils.CompletableFutureUtils;
import software.amazon.awssdk.utils.Logger;
import software.amazon.awssdk.utils.StringUtils;
import software.amazon.awssdk.utils.Validate;

@SdkInternalApi
public class DownloadDirectoryHelper {
    private static final Logger log = Logger.loggerFor(S3TransferManager.class);
    private final TransferManagerConfiguration transferConfiguration;
    private final Function<DownloadFileRequest, FileDownload> downloadFileFunction;
    private final ListObjectsHelper listObjectsHelper;

    public DownloadDirectoryHelper(TransferManagerConfiguration transferConfiguration, ListObjectsHelper listObjectsHelper, Function<DownloadFileRequest, FileDownload> downloadFileFunction) {
        this.transferConfiguration = transferConfiguration;
        this.downloadFileFunction = downloadFileFunction;
        this.listObjectsHelper = listObjectsHelper;
    }

    public DirectoryDownload downloadDirectory(DownloadDirectoryRequest downloadDirectoryRequest) {
        CompletableFuture<CompletedDirectoryDownload> returnFuture = new CompletableFuture<CompletedDirectoryDownload>();
        CompletableFuture.runAsync(() -> this.doDownloadDirectory(returnFuture, downloadDirectoryRequest), this.transferConfiguration.option(TransferConfigurationOption.EXECUTOR)).whenComplete((r, t2) -> {
            if (t2 != null) {
                returnFuture.completeExceptionally((Throwable)t2);
            }
        });
        return new DefaultDirectoryDownload(returnFuture);
    }

    private static void validateDirectoryIfExists(Path directory) {
        if (Files.exists(directory, new LinkOption[0])) {
            Validate.isTrue((boolean)Files.isDirectory(directory, new LinkOption[0]), (String)"The destination directory provided (%s) is not a directory", (Object[])new Object[]{directory});
        }
    }

    private void doDownloadDirectory(CompletableFuture<CompletedDirectoryDownload> returnFuture, DownloadDirectoryRequest downloadDirectoryRequest) {
        DownloadDirectoryHelper.validateDirectoryIfExists(downloadDirectoryRequest.destination());
        String bucket = downloadDirectoryRequest.bucket();
        ListObjectsV2Request request = (ListObjectsV2Request)((Object)((ListObjectsV2Request.Builder)ListObjectsV2Request.builder().bucket(bucket).prefix("").applyMutation(downloadDirectoryRequest.listObjectsRequestTransformer())).build());
        ConcurrentLinkedQueue<FailedFileDownload> failedFileDownloads = new ConcurrentLinkedQueue<FailedFileDownload>();
        CompletableFuture<Void> allOfFutures = new CompletableFuture<Void>();
        AsyncBufferingSubscriber<S3Object> asyncBufferingSubscriber = new AsyncBufferingSubscriber<S3Object>(this.downloadSingleFile(downloadDirectoryRequest, request, failedFileDownloads), allOfFutures, this.transferConfiguration.option(TransferConfigurationOption.DIRECTORY_TRANSFER_MAX_CONCURRENCY));
        this.listObjectsHelper.listS3ObjectsRecursively(request).filter((Predicate)downloadDirectoryRequest.filter()).subscribe(asyncBufferingSubscriber);
        CompletableFutureUtils.forwardExceptionTo(returnFuture, allOfFutures);
        allOfFutures.whenComplete((r, t2) -> {
            if (t2 != null) {
                returnFuture.completeExceptionally(SdkClientException.create((String)"Failed to send request", (Throwable)t2));
            } else {
                returnFuture.complete(CompletedDirectoryDownload.builder().failedTransfers(failedFileDownloads).build());
            }
        });
    }

    private Function<S3Object, CompletableFuture<?>> downloadSingleFile(DownloadDirectoryRequest downloadDirectoryRequest, ListObjectsV2Request listRequest, Queue<FailedFileDownload> failedFileDownloads) {
        return s3Object -> this.doDownloadSingleFile(downloadDirectoryRequest, (Collection<FailedFileDownload>)failedFileDownloads, listRequest, (S3Object)s3Object);
    }

    private Path determineDestinationPath(DownloadDirectoryRequest downloadDirectoryRequest, ListObjectsV2Request listRequest, S3Object s3Object) {
        FileSystem fileSystem = downloadDirectoryRequest.destination().getFileSystem();
        String delimiter = listRequest.delimiter() == null ? "/" : listRequest.delimiter();
        String key = DirectoryHelperUtils.normalizeKey(listRequest.prefix(), s3Object.key(), delimiter);
        String relativePath = DownloadDirectoryHelper.getRelativePath(fileSystem, delimiter, key);
        Path destinationPath = downloadDirectoryRequest.destination().resolve(relativePath);
        this.validatePath(downloadDirectoryRequest.destination(), destinationPath, s3Object.key());
        return destinationPath;
    }

    private void validatePath(Path destinationDirectory, Path targetPath, String key) {
        if (!targetPath.toAbsolutePath().normalize().startsWith(destinationDirectory.toAbsolutePath().normalize())) {
            throw SdkClientException.create((String)("Cannot download key " + key + ", its relative path resolves outside the parent directory."));
        }
    }

    private CompletableFuture<CompletedFileDownload> doDownloadSingleFile(DownloadDirectoryRequest downloadDirectoryRequest, Collection<FailedFileDownload> failedFileDownloads, ListObjectsV2Request listRequest, S3Object s3Object) {
        Path destinationPath = this.determineDestinationPath(downloadDirectoryRequest, listRequest, s3Object);
        DownloadFileRequest downloadFileRequest = DownloadDirectoryHelper.downloadFileRequest(downloadDirectoryRequest, s3Object, destinationPath);
        try {
            log.debug(() -> "Sending download request " + downloadFileRequest);
            DownloadDirectoryHelper.createParentDirectoriesIfNeeded(destinationPath);
            CompletableFuture<CompletedFileDownload> executionFuture = this.downloadFileFunction.apply(downloadFileRequest).completionFuture();
            CompletionStage future = executionFuture.whenComplete((r, t2) -> {
                if (t2 != null) {
                    failedFileDownloads.add((FailedFileDownload)FailedFileDownload.builder().exception(t2 instanceof CompletionException ? t2.getCause() : t2).request(downloadFileRequest).build());
                }
            });
            CompletableFutureUtils.forwardExceptionTo((CompletableFuture)future, executionFuture);
            return future;
        }
        catch (Throwable throwable) {
            failedFileDownloads.add((FailedFileDownload)FailedFileDownload.builder().exception(throwable).request(downloadFileRequest).build());
            return CompletableFutureUtils.failedFuture((Throwable)throwable);
        }
    }

    private static String getRelativePath(FileSystem fileSystem, String delimiter, String key) {
        if (delimiter == null) {
            return key;
        }
        if (fileSystem.getSeparator().equals(delimiter)) {
            return key;
        }
        return StringUtils.replace((String)key, (String)delimiter, (String)fileSystem.getSeparator());
    }

    private static DownloadFileRequest downloadFileRequest(DownloadDirectoryRequest downloadDirectoryRequest, S3Object s3Object, Path destinationPath) {
        GetObjectRequest getObjectRequest = (GetObjectRequest)((Object)GetObjectRequest.builder().bucket(downloadDirectoryRequest.bucket()).key(s3Object.key()).build());
        return (DownloadFileRequest)((DownloadFileRequest.Builder)DownloadFileRequest.builder().destination(destinationPath).getObjectRequest(getObjectRequest).applyMutation(downloadDirectoryRequest.downloadFileRequestTransformer())).build();
    }

    private static void createParentDirectoriesIfNeeded(Path destinationPath) {
        Path parentDirectory = destinationPath.getParent();
        try {
            if (parentDirectory != null) {
                Files.createDirectories(parentDirectory, new FileAttribute[0]);
            }
        }
        catch (IOException e) {
            throw SdkClientException.create((String)("Failed to create parent directories for " + destinationPath), (Throwable)e);
        }
    }
}

