/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.fs.s3a.impl;

import java.io.BufferedInputStream;
import java.io.ByteArrayInputStream;
import java.io.Closeable;
import java.io.File;
import java.io.FileInputStream;
import java.io.InputStream;
import java.io.UncheckedIOException;
import java.nio.ByteBuffer;
import java.time.LocalDateTime;
import java.util.Objects;
import java.util.function.Supplier;
import javax.annotation.Nullable;
import org.apache.hadoop.classification.VisibleForTesting;
import org.apache.hadoop.fs.store.ByteBufferInputStream;
import org.apache.hadoop.io.IOUtils;
import org.apache.hadoop.util.Preconditions;
import org.apache.hadoop.util.functional.FunctionalIO;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.http.ContentStreamProvider;

public final class UploadContentProviders {
    public static final Logger LOG = LoggerFactory.getLogger(UploadContentProviders.class);

    private UploadContentProviders() {
    }

    public static BaseContentProvider<BufferedInputStream> fileContentProvider(File file, long offset, int size) {
        return new FileWithOffsetContentProvider(file, offset, size);
    }

    public static BaseContentProvider<BufferedInputStream> fileContentProvider(File file, long offset, int size, Supplier<Boolean> isOpen) {
        return new FileWithOffsetContentProvider(file, offset, size, isOpen);
    }

    public static BaseContentProvider<ByteBufferInputStream> byteBufferContentProvider(ByteBuffer byteBuffer, int size) {
        return new ByteBufferContentProvider(byteBuffer, size);
    }

    public static BaseContentProvider<ByteBufferInputStream> byteBufferContentProvider(ByteBuffer byteBuffer, int size, @Nullable Supplier<Boolean> isOpen) {
        return new ByteBufferContentProvider(byteBuffer, size, isOpen);
    }

    public static BaseContentProvider<ByteArrayInputStream> byteArrayContentProvider(byte[] bytes, int offset, int size) {
        return new ByteArrayContentProvider(bytes, offset, size);
    }

    public static BaseContentProvider<ByteArrayInputStream> byteArrayContentProvider(byte[] bytes, int offset, int size, @Nullable Supplier<Boolean> isOpen) {
        return new ByteArrayContentProvider(bytes, offset, size, isOpen);
    }

    public static BaseContentProvider<ByteArrayInputStream> byteArrayContentProvider(byte[] bytes) {
        return UploadContentProviders.byteArrayContentProvider(bytes, 0, bytes.length);
    }

    public static BaseContentProvider<ByteArrayInputStream> byteArrayContentProvider(byte[] bytes, @Nullable Supplier<Boolean> isOpen) {
        return UploadContentProviders.byteArrayContentProvider(bytes, 0, bytes.length, isOpen);
    }

    private static final class ByteArrayContentProvider
    extends BaseContentProvider<ByteArrayInputStream> {
        private final byte[] bytes;
        private final int offset;

        private ByteArrayContentProvider(byte[] bytes, int offset, int size) {
            this(bytes, offset, size, (Supplier<Boolean>)null);
        }

        private ByteArrayContentProvider(byte[] bytes, int offset, int size, Supplier<Boolean> isOpen) {
            super(size, isOpen);
            this.bytes = bytes;
            this.offset = offset;
            Preconditions.checkArgument(offset >= 0, "Offset is negative: %s", offset);
            int length = bytes.length;
            Preconditions.checkArgument(offset + size <= length, "Data to read [%d-%d] is past end of array %s", offset, offset + size, length);
        }

        @Override
        protected ByteArrayInputStream createNewStream() {
            return new ByteArrayInputStream(this.bytes, this.offset, this.getSize());
        }

        @Override
        public String toString() {
            return "ByteArrayContentProvider{buffer with length=" + this.bytes.length + ", offset=" + this.offset + "} " + super.toString();
        }
    }

    private static final class ByteBufferContentProvider
    extends BaseContentProvider<ByteBufferInputStream> {
        private final ByteBuffer blockBuffer;
        private final int initialPosition;

        private ByteBufferContentProvider(ByteBuffer blockBuffer, int size) {
            this(blockBuffer, size, (Supplier<Boolean>)null);
        }

        private ByteBufferContentProvider(ByteBuffer blockBuffer, int size, @Nullable Supplier<Boolean> isOpen) {
            super(size, isOpen);
            this.blockBuffer = blockBuffer;
            this.initialPosition = blockBuffer.position();
        }

        @Override
        protected ByteBufferInputStream createNewStream() {
            this.blockBuffer.limit(this.initialPosition);
            this.blockBuffer.position(0);
            return new ByteBufferInputStream(this.getSize(), this.blockBuffer);
        }

        @Override
        public String toString() {
            return "ByteBufferContentProvider{blockBuffer=" + this.blockBuffer + ", initialPosition=" + this.initialPosition + "} " + super.toString();
        }
    }

    private static final class FileWithOffsetContentProvider
    extends BaseContentProvider<BufferedInputStream> {
        private final File file;
        private final long offset;

        private FileWithOffsetContentProvider(File file, long offset, int size, @Nullable Supplier<Boolean> isOpen) {
            super(size, isOpen);
            this.file = Objects.requireNonNull(file);
            Preconditions.checkArgument(offset >= 0L, "Offset is negative: %s", offset);
            this.offset = offset;
        }

        private FileWithOffsetContentProvider(File file, long offset, int size) {
            this(file, offset, size, (Supplier<Boolean>)null);
        }

        @Override
        protected BufferedInputStream createNewStream() throws UncheckedIOException {
            FileInputStream fis = FunctionalIO.uncheckIOExceptions(() -> {
                FileInputStream f = new FileInputStream(this.file);
                f.getChannel().position(this.offset);
                return f;
            });
            return this.setCurrentStream(new BufferedInputStream(fis));
        }

        @Override
        public String toString() {
            return "FileWithOffsetContentProvider{file=" + this.file + ", offset=" + this.offset + "} " + super.toString();
        }
    }

    @VisibleForTesting
    public static abstract class BaseContentProvider<T extends InputStream>
    implements ContentStreamProvider,
    Closeable {
        private final int size;
        private final Supplier<Boolean> isOpen;
        private int streamCreationCount;
        private T currentStream;
        private final LocalDateTime startTime;

        protected BaseContentProvider(int size) {
            this(size, null);
        }

        protected BaseContentProvider(int size, @Nullable Supplier<Boolean> isOpen) {
            Preconditions.checkArgument(size >= 0, "size is negative: %s", size);
            this.size = size;
            this.isOpen = isOpen;
            this.startTime = LocalDateTime.now();
        }

        private void checkOpen() {
            Preconditions.checkState(this.isOpen == null || this.isOpen.get() != false, "Stream is closed: %s", this);
        }

        @Override
        public void close() {
            IOUtils.cleanupWithLogger(LOG, new Closeable[]{this.getCurrentStream()});
            this.setCurrentStream(null);
        }

        public final InputStream newStream() {
            this.close();
            this.checkOpen();
            ++this.streamCreationCount;
            if (this.streamCreationCount == 2) {
                LOG.debug("Stream recreated: {}", (Object)this);
            }
            return this.setCurrentStream(this.createNewStream());
        }

        protected abstract T createNewStream();

        public int getStreamCreationCount() {
            return this.streamCreationCount;
        }

        public int getSize() {
            return this.size;
        }

        public LocalDateTime getStartTime() {
            return this.startTime;
        }

        protected T getCurrentStream() {
            return this.currentStream;
        }

        protected T setCurrentStream(T stream) {
            this.currentStream = stream;
            return stream;
        }

        public String toString() {
            return "BaseContentProvider{size=" + this.size + ", initiated at " + this.startTime + ", streamCreationCount=" + this.streamCreationCount + ", currentStream=" + this.currentStream + '}';
        }
    }
}

