/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.fs.s3a;

import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InterruptedIOException;
import java.io.OutputStream;
import java.time.Duration;
import java.time.Instant;
import java.util.ArrayList;
import java.util.Collections;
import java.util.EnumSet;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Objects;
import java.util.StringJoiner;
import java.util.concurrent.CancellationException;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Future;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicInteger;
import javax.annotation.Nonnull;
import org.apache.commons.lang3.StringUtils;
import org.apache.hadoop.classification.InterfaceAudience;
import org.apache.hadoop.classification.InterfaceStability;
import org.apache.hadoop.classification.VisibleForTesting;
import org.apache.hadoop.fs.Abortable;
import org.apache.hadoop.fs.ClosedIOException;
import org.apache.hadoop.fs.StreamCapabilities;
import org.apache.hadoop.fs.Syncable;
import org.apache.hadoop.fs.s3a.S3ADataBlocks;
import org.apache.hadoop.fs.s3a.S3AUtils;
import org.apache.hadoop.fs.s3a.Statistic;
import org.apache.hadoop.fs.s3a.WriteOperationHelper;
import org.apache.hadoop.fs.s3a.WriteOperations;
import org.apache.hadoop.fs.s3a.commit.PutTracker;
import org.apache.hadoop.fs.s3a.impl.ProgressListener;
import org.apache.hadoop.fs.s3a.impl.ProgressListenerEvent;
import org.apache.hadoop.fs.s3a.impl.PutObjectOptions;
import org.apache.hadoop.fs.s3a.impl.write.WriteObjectFlags;
import org.apache.hadoop.fs.s3a.statistics.BlockOutputStreamStatistics;
import org.apache.hadoop.fs.s3a.statistics.impl.EmptyS3AStatisticsContext;
import org.apache.hadoop.fs.statistics.DurationTracker;
import org.apache.hadoop.fs.statistics.IOStatistics;
import org.apache.hadoop.fs.statistics.IOStatisticsAggregator;
import org.apache.hadoop.fs.statistics.IOStatisticsLogging;
import org.apache.hadoop.fs.statistics.IOStatisticsSource;
import org.apache.hadoop.fs.statistics.impl.IOStatisticsBinding;
import org.apache.hadoop.fs.store.LogExactlyOnce;
import org.apache.hadoop.io.IOUtils;
import org.apache.hadoop.thirdparty.com.google.common.util.concurrent.ListenableFuture;
import org.apache.hadoop.thirdparty.com.google.common.util.concurrent.ListeningExecutorService;
import org.apache.hadoop.thirdparty.com.google.common.util.concurrent.MoreExecutors;
import org.apache.hadoop.util.Preconditions;
import org.apache.hadoop.util.Progressable;
import org.apache.hadoop.util.functional.FutureIO;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.core.exception.SdkException;
import software.amazon.awssdk.core.sync.RequestBody;
import software.amazon.awssdk.services.s3.model.CompletedPart;
import software.amazon.awssdk.services.s3.model.PutObjectRequest;
import software.amazon.awssdk.services.s3.model.UploadPartRequest;
import software.amazon.awssdk.services.s3.model.UploadPartResponse;

@InterfaceAudience.Private
@InterfaceStability.Unstable
class S3ABlockOutputStream
extends OutputStream
implements StreamCapabilities,
IOStatisticsSource,
Syncable,
Abortable {
    private static final Logger LOG = LoggerFactory.getLogger(S3ABlockOutputStream.class);
    private static final String E_NOT_SYNCABLE = "S3A streams are not Syncable. See HADOOP-17597.";
    private static final Duration TIME_TO_AWAIT_CANCEL_COMPLETION = Duration.ofSeconds(15L);
    private final String key;
    private final long blockSize;
    private final IOStatistics iostatistics;
    private final BlockOutputStreamBuilder builder;
    private long bytesSubmitted;
    private final ProgressListener progressListener;
    private final ListeningExecutorService executorService;
    private final S3ADataBlocks.BlockFactory blockFactory;
    private final byte[] singleCharWrite = new byte[1];
    private MultiPartUpload multiPartUpload;
    private final AtomicBoolean closed = new AtomicBoolean(false);
    private S3ADataBlocks.DataBlock activeBlock;
    private long blockCount = 0L;
    private final BlockOutputStreamStatistics statistics;
    private final WriteOperations writeOperationHelper;
    private final PutTracker putTracker;
    private final boolean downgradeSyncableExceptions;
    private static final LogExactlyOnce WARN_ON_SYNCABLE = new LogExactlyOnce(LOG);
    private final boolean isCSEEnabled;
    private final IOStatisticsAggregator threadIOStatisticsAggregator;
    private final boolean isMultipartUploadEnabled;
    private final EnumSet<WriteObjectFlags> writeObjectFlags;

    S3ABlockOutputStream(BlockOutputStreamBuilder builder) throws IOException {
        builder.validate();
        this.builder = builder;
        this.key = builder.key;
        this.blockFactory = builder.blockFactory;
        this.statistics = builder.statistics;
        this.iostatistics = this.statistics.getIOStatistics();
        this.writeOperationHelper = builder.writeOperations;
        this.putTracker = builder.putTracker;
        this.writeObjectFlags = builder.putOptions.getWriteObjectFlags();
        this.executorService = MoreExecutors.listeningDecorator(builder.executorService);
        this.multiPartUpload = null;
        Progressable progress = builder.progress;
        this.progressListener = progress instanceof ProgressListener ? (ProgressListener)((Object)progress) : new ProgressableListener(progress);
        this.downgradeSyncableExceptions = builder.downgradeSyncableExceptions;
        this.isMultipartUploadEnabled = builder.isMultipartUploadEnabled;
        long l = this.blockSize = this.isMultipartUploadEnabled ? builder.blockSize : -1L;
        if (this.putTracker.initialize()) {
            LOG.debug("Put tracker requests multipart upload");
            this.initMultipartUpload();
        } else if (this.writeObjectFlags.contains((Object)WriteObjectFlags.CreateMultipart)) {
            LOG.debug("Multipart initiated from createFile() options");
            this.initMultipartUpload();
        }
        this.isCSEEnabled = builder.isCSEEnabled;
        this.threadIOStatisticsAggregator = builder.ioStatisticsAggregator;
        this.createBlockIfNeeded();
        LOG.debug("Initialized S3ABlockOutputStream for {} output to {}", (Object)this.key, (Object)this.activeBlock);
    }

    private synchronized S3ADataBlocks.DataBlock createBlockIfNeeded() throws IOException {
        if (this.activeBlock == null) {
            ++this.blockCount;
            if (this.blockCount >= 10000L) {
                LOG.error("Number of partitions in stream exceeds limit for S3: 10000 write may fail.");
            }
            this.activeBlock = this.blockFactory.create(this.blockCount, this.blockSize, this.statistics);
        }
        return this.activeBlock;
    }

    private synchronized S3ADataBlocks.DataBlock getActiveBlock() {
        return this.activeBlock;
    }

    private synchronized boolean hasActiveBlock() {
        return this.activeBlock != null;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void clearActiveBlock() {
        if (this.activeBlock != null) {
            LOG.debug("Clearing active block");
        }
        S3ABlockOutputStream s3ABlockOutputStream = this;
        synchronized (s3ABlockOutputStream) {
            this.activeBlock = null;
        }
    }

    @VisibleForTesting
    void checkOpen() throws ClosedIOException {
        if (this.closed.get()) {
            throw new ClosedIOException(this.key, "Stream is closed:  " + this);
        }
    }

    @Override
    public synchronized void flush() throws IOException {
        try {
            this.checkOpen();
        }
        catch (ClosedIOException e) {
            LOG.warn("Stream closed: {}", (Object)e.getMessage());
            return;
        }
        S3ADataBlocks.DataBlock dataBlock = this.getActiveBlock();
        if (dataBlock != null) {
            dataBlock.flush();
        }
    }

    @Override
    public synchronized void write(int b) throws IOException {
        this.singleCharWrite[0] = (byte)b;
        this.write(this.singleCharWrite, 0, 1);
    }

    @Override
    public synchronized void write(@Nonnull byte[] source, int offset, int len) throws IOException {
        S3ADataBlocks.validateWriteArgs(source, offset, len);
        this.checkOpen();
        if (len == 0) {
            return;
        }
        this.statistics.writeBytes(len);
        S3ADataBlocks.DataBlock block = this.createBlockIfNeeded();
        int written = block.write(source, offset, len);
        if (!this.isMultipartUploadEnabled) {
            return;
        }
        int remainingCapacity = (int)block.remainingCapacity();
        if (written < len) {
            LOG.debug("writing more data than block has capacity -triggering upload");
            this.uploadCurrentBlock(false);
            this.write(source, offset + written, len - written);
        } else if (remainingCapacity == 0 && !this.isCSEEnabled) {
            this.uploadCurrentBlock(false);
        }
    }

    private synchronized void uploadCurrentBlock(boolean isLast) throws IOException {
        Preconditions.checkState(this.hasActiveBlock(), "No active block");
        LOG.debug("Writing block # {}", (Object)this.blockCount);
        this.initMultipartUpload();
        try {
            this.multiPartUpload.uploadBlockAsync(this.getActiveBlock(), isLast);
            this.bytesSubmitted += this.getActiveBlock().dataSize();
        }
        finally {
            this.clearActiveBlock();
        }
    }

    private void initMultipartUpload() throws IOException {
        Preconditions.checkState(this.isMultipartUploadEnabled, "multipart upload is disabled");
        if (this.multiPartUpload == null) {
            LOG.debug("Initiating Multipart upload");
            this.multiPartUpload = new MultiPartUpload(this.key);
        }
    }

    @Override
    public void close() throws IOException {
        if (this.closed.getAndSet(true)) {
            LOG.debug("Ignoring close() as stream is already closed");
            return;
        }
        this.progressListener.progressChanged(ProgressListenerEvent.CLOSE_EVENT, 0L);
        S3ADataBlocks.DataBlock block = this.getActiveBlock();
        boolean hasBlock = this.hasActiveBlock();
        LOG.debug("{}: Closing block #{}: current block= {}", this, this.blockCount, hasBlock ? block : "(none)");
        long bytes = 0L;
        try {
            if (this.multiPartUpload == null) {
                if (hasBlock) {
                    this.bytesSubmitted = bytes = this.putObject();
                }
            } else {
                if (hasBlock && (block.hasData() || this.multiPartUpload.getPartsSubmitted() == 0)) {
                    this.uploadCurrentBlock(true);
                }
                List partETags = this.multiPartUpload.waitForAllPartUploads();
                bytes = this.bytesSubmitted;
                String uploadId = this.multiPartUpload.getUploadId();
                LOG.debug("Multipart upload to {} ID {} containing {} blocks", this.key, uploadId, partETags.size());
                if (this.putTracker.aboutToComplete(uploadId, partETags, bytes, this.iostatistics)) {
                    this.multiPartUpload.complete(partETags);
                } else {
                    LOG.info("File {} will be visible when the job is committed", (Object)this.key);
                }
            }
            if (!this.putTracker.outputImmediatelyVisible()) {
                this.statistics.commitUploaded(bytes);
            }
            LOG.debug("Upload complete to {} by {}", (Object)this.key, (Object)this.writeOperationHelper);
        }
        catch (IOException ioe) {
            this.maybeAbortMultipart();
            this.writeOperationHelper.writeFailed(ioe);
            throw ioe;
        }
        catch (CancellationException e) {
            this.maybeAbortMultipart();
            this.writeOperationHelper.writeFailed(e);
            throw (IOException)new InterruptedIOException(e.getMessage()).initCause(e);
        }
        finally {
            this.cleanupOnClose();
        }
        this.writeOperationHelper.writeSuccessful(bytes);
    }

    private synchronized void cleanupOnClose() {
        IOUtils.cleanupWithLogger(LOG, this.getActiveBlock(), this.blockFactory);
        this.mergeThreadIOStatistics(this.statistics.getIOStatistics());
        LOG.debug("Statistics: {}", (Object)this.statistics);
        IOUtils.cleanupWithLogger(LOG, this.statistics);
        this.clearActiveBlock();
    }

    private void mergeThreadIOStatistics(IOStatistics streamStatistics) {
        this.getThreadIOStatistics().aggregate(streamStatistics);
    }

    private synchronized IOException maybeAbortMultipart() {
        if (this.multiPartUpload != null) {
            try {
                IOException iOException = this.multiPartUpload.abort();
                return iOException;
            }
            finally {
                this.multiPartUpload = null;
            }
        }
        return null;
    }

    /*
     * Loose catch block
     */
    @Override
    public Abortable.AbortableResult abort() {
        if (this.closed.getAndSet(true)) {
            LOG.debug("Ignoring abort() as stream is already closed");
            return new AbortableResultImpl(true, null);
        }
        try {
            try (DurationTracker d = this.statistics.trackDuration(Statistic.INVOCATION_ABORT.getSymbol());){
                IOException anyCleanupException = this.maybeAbortMultipart();
                AbortableResultImpl abortableResultImpl = new AbortableResultImpl(anyCleanupException instanceof FileNotFoundException, anyCleanupException);
                return abortableResultImpl;
            }
            {
                catch (Throwable throwable) {
                    throw throwable;
                }
            }
        }
        finally {
            this.cleanupOnClose();
        }
    }

    /*
     * Loose catch block
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    private long putObject() throws IOException {
        LOG.debug("Executing regular upload for {}", (Object)this.writeOperationHelper);
        S3ADataBlocks.DataBlock block = this.getActiveBlock();
        long size = block.dataSize();
        S3ADataBlocks.BlockUploadData uploadData = block.startUpload();
        PutObjectRequest putObjectRequest = this.writeOperationHelper.createPutObjectRequest(this.key, uploadData.getSize(), this.builder.putOptions);
        this.clearActiveBlock();
        BlockUploadProgress progressCallback = new BlockUploadProgress(block, this.progressListener, this.now());
        this.statistics.blockUploadQueued(size);
        try {
            progressCallback.progressChanged(ProgressListenerEvent.PUT_STARTED_EVENT);
            this.writeOperationHelper.putObject(putObjectRequest, this.builder.putOptions, uploadData, this.statistics);
            progressCallback.progressChanged(ProgressListenerEvent.REQUEST_BYTE_TRANSFER_EVENT);
            progressCallback.progressChanged(ProgressListenerEvent.PUT_COMPLETED_EVENT);
        }
        catch (InterruptedIOException ioe) {
            try {
                progressCallback.progressChanged(ProgressListenerEvent.PUT_INTERRUPTED_EVENT);
                throw ioe;
                catch (IOException ioe2) {
                    progressCallback.progressChanged(ProgressListenerEvent.PUT_FAILED_EVENT);
                    throw ioe2;
                }
            }
            catch (Throwable throwable) {
                IOUtils.cleanupWithLogger(LOG, uploadData, block);
                throw throwable;
            }
        }
        IOUtils.cleanupWithLogger(LOG, uploadData, block);
        return size;
    }

    public String toString() {
        StringBuilder sb = new StringBuilder("S3ABlockOutputStream{");
        sb.append(this.writeOperationHelper.toString());
        sb.append(", blockSize=").append(this.blockSize);
        sb.append(", isMultipartUploadEnabled=").append(this.isMultipartUploadEnabled);
        S3ADataBlocks.DataBlock block = this.activeBlock;
        if (block != null) {
            sb.append(", activeBlock=").append(block);
        }
        sb.append(" Statistics=").append(IOStatisticsLogging.ioStatisticsSourceToString(this));
        sb.append('}');
        return sb.toString();
    }

    private void incrementWriteOperations() {
        this.writeOperationHelper.incrementWriteOperations();
    }

    private Instant now() {
        return Instant.now();
    }

    BlockOutputStreamStatistics getStatistics() {
        return this.statistics;
    }

    @Override
    public boolean hasCapability(String capability) {
        String cap;
        switch (cap = capability.toLowerCase(Locale.ENGLISH)) {
            case "fs.s3a.capability.magic.output.stream": 
            case "s3a:magic.output.stream": {
                return !this.putTracker.outputImmediatelyVisible();
            }
            case "hflush": 
            case "hsync": {
                return false;
            }
            case "iostatistics": {
                return true;
            }
            case "fs.capability.outputstream.abortable": {
                return true;
            }
            case "fs.capability.iocontext.supported": {
                return true;
            }
        }
        for (WriteObjectFlags flag : this.writeObjectFlags) {
            if (!flag.hasKey(cap)) continue;
            return true;
        }
        return false;
    }

    @Override
    public void hflush() throws IOException {
        this.statistics.hflushInvoked();
        LOG.debug("Hflush invoked");
    }

    @Override
    public void hsync() throws IOException {
        this.statistics.hsyncInvoked();
        this.handleSyncableInvocation();
    }

    private void handleSyncableInvocation() {
        UnsupportedOperationException ex = new UnsupportedOperationException(E_NOT_SYNCABLE);
        if (!this.downgradeSyncableExceptions) {
            throw ex;
        }
        WARN_ON_SYNCABLE.warn("Application invoked the Syncable API against stream writing to {}. This is Unsupported", this.key);
        LOG.debug("Downgrading Syncable call", ex);
    }

    @Override
    public IOStatistics getIOStatistics() {
        return this.iostatistics;
    }

    protected IOStatisticsAggregator getThreadIOStatistics() {
        return this.threadIOStatisticsAggregator;
    }

    public static BlockOutputStreamBuilder builder() {
        return new BlockOutputStreamBuilder();
    }

    public static final class BlockOutputStreamBuilder {
        private String key;
        private ExecutorService executorService;
        private Progressable progress;
        private long blockSize;
        private S3ADataBlocks.BlockFactory blockFactory;
        private BlockOutputStreamStatistics statistics = EmptyS3AStatisticsContext.EMPTY_BLOCK_OUTPUT_STREAM_STATISTICS;
        private WriteOperations writeOperations;
        private PutTracker putTracker;
        private boolean downgradeSyncableExceptions;
        private boolean isCSEEnabled;
        private PutObjectOptions putOptions;
        private IOStatisticsAggregator ioStatisticsAggregator;
        private boolean isMultipartUploadEnabled;

        private BlockOutputStreamBuilder() {
        }

        public void validate() {
            Objects.requireNonNull(this.key, "null key");
            Objects.requireNonNull(this.executorService, "null executorService");
            Objects.requireNonNull(this.blockFactory, "null blockFactory");
            Objects.requireNonNull(this.statistics, "null statistics");
            Objects.requireNonNull(this.writeOperations, "null writeOperationHelper");
            Objects.requireNonNull(this.putTracker, "null putTracker");
            Objects.requireNonNull(this.putOptions, "null putOptions");
            Preconditions.checkArgument(this.blockSize >= 0x500000L, "Block size is too small: %s", this.blockSize);
            Objects.requireNonNull(this.ioStatisticsAggregator, "null ioStatisticsAggregator");
        }

        public BlockOutputStreamBuilder withKey(String value) {
            this.key = value;
            return this;
        }

        public BlockOutputStreamBuilder withExecutorService(ExecutorService value) {
            this.executorService = value;
            return this;
        }

        public BlockOutputStreamBuilder withProgress(Progressable value) {
            this.progress = value;
            return this;
        }

        public BlockOutputStreamBuilder withBlockSize(long value) {
            this.blockSize = value;
            return this;
        }

        public BlockOutputStreamBuilder withBlockFactory(S3ADataBlocks.BlockFactory value) {
            this.blockFactory = value;
            return this;
        }

        public BlockOutputStreamBuilder withStatistics(BlockOutputStreamStatistics value) {
            this.statistics = value;
            return this;
        }

        public BlockOutputStreamBuilder withWriteOperations(WriteOperationHelper value) {
            this.writeOperations = value;
            return this;
        }

        public BlockOutputStreamBuilder withPutTracker(PutTracker value) {
            this.putTracker = value;
            return this;
        }

        public BlockOutputStreamBuilder withDowngradeSyncableExceptions(boolean value) {
            this.downgradeSyncableExceptions = value;
            return this;
        }

        public BlockOutputStreamBuilder withCSEEnabled(boolean value) {
            this.isCSEEnabled = value;
            return this;
        }

        public BlockOutputStreamBuilder withPutOptions(PutObjectOptions value) {
            this.putOptions = value;
            return this;
        }

        public BlockOutputStreamBuilder withIOStatisticsAggregator(IOStatisticsAggregator value) {
            this.ioStatisticsAggregator = value;
            return this;
        }

        public BlockOutputStreamBuilder withMultipartEnabled(boolean value) {
            this.isMultipartUploadEnabled = value;
            return this;
        }
    }

    private static final class ProgressableListener
    implements ProgressListener {
        private final Progressable progress;

        ProgressableListener(Progressable progress) {
            this.progress = progress;
        }

        @Override
        public void progressChanged(ProgressListenerEvent eventType, long bytesTransferred) {
            if (this.progress != null) {
                this.progress.progress();
            }
        }
    }

    private final class BlockUploadProgress {
        private final S3ADataBlocks.DataBlock block;
        private final ProgressListener nextListener;
        private final Instant transferQueueTime;
        private Instant transferStartTime;
        private long size;

        private BlockUploadProgress(S3ADataBlocks.DataBlock block, ProgressListener nextListener, Instant transferQueueTime) {
            this.block = block;
            this.transferQueueTime = transferQueueTime;
            this.size = block.dataSize();
            this.nextListener = nextListener;
            this.transferStartTime = S3ABlockOutputStream.this.now();
        }

        public void progressChanged(ProgressListenerEvent eventType) {
            switch (eventType) {
                case PUT_STARTED_EVENT: 
                case TRANSFER_PART_STARTED_EVENT: {
                    this.transferStartTime = S3ABlockOutputStream.this.now();
                    S3ABlockOutputStream.this.statistics.blockUploadStarted(Duration.between(this.transferQueueTime, this.transferStartTime), this.size);
                    S3ABlockOutputStream.this.incrementWriteOperations();
                    break;
                }
                case TRANSFER_PART_COMPLETED_EVENT: 
                case PUT_COMPLETED_EVENT: {
                    S3ABlockOutputStream.this.statistics.blockUploadCompleted(Duration.between(this.transferStartTime, S3ABlockOutputStream.this.now()), this.size);
                    S3ABlockOutputStream.this.statistics.bytesTransferred(this.size);
                    break;
                }
                case TRANSFER_PART_FAILED_EVENT: 
                case PUT_FAILED_EVENT: 
                case PUT_INTERRUPTED_EVENT: {
                    S3ABlockOutputStream.this.statistics.blockUploadFailed(Duration.between(this.transferStartTime, S3ABlockOutputStream.this.now()), this.size);
                    LOG.warn("Transfer failure of block {}", (Object)this.block);
                    break;
                }
            }
            if (this.nextListener != null) {
                this.nextListener.progressChanged(eventType, this.size);
            }
        }
    }

    private class MultiPartUpload {
        private final String uploadId;
        private final List<Future<CompletedPart>> partETagsFutures = Collections.synchronizedList(new ArrayList());
        private final Map<Integer, S3ADataBlocks.DataBlock> blocksToClose = new ConcurrentHashMap<Integer, S3ADataBlocks.DataBlock>();
        private int partsSubmitted;
        private int partsUploaded;
        private long bytesSubmitted;
        private final AtomicBoolean uploadAborted = new AtomicBoolean(false);
        private IOException blockUploadFailure;

        MultiPartUpload(String key) throws IOException {
            this.uploadId = IOStatisticsBinding.trackDuration(S3ABlockOutputStream.this.statistics, Statistic.OBJECT_MULTIPART_UPLOAD_INITIATED.getSymbol(), () -> S3ABlockOutputStream.this.writeOperationHelper.initiateMultiPartUpload(key, S3ABlockOutputStream.this.builder.putOptions));
            LOG.debug("Initiated multi-part upload for {} with id '{}'", (Object)S3ABlockOutputStream.this.writeOperationHelper, (Object)this.uploadId);
            S3ABlockOutputStream.this.progressListener.progressChanged(ProgressListenerEvent.TRANSFER_MULTIPART_INITIATED_EVENT, 0L);
        }

        public int getPartsSubmitted() {
            return this.partsSubmitted;
        }

        public int getPartsUploaded() {
            return this.partsUploaded;
        }

        public String getUploadId() {
            return this.uploadId;
        }

        public long getBytesSubmitted() {
            return this.bytesSubmitted;
        }

        public void noteUploadFailure(IOException e) {
            if (this.blockUploadFailure == null) {
                this.blockUploadFailure = e;
            }
        }

        public void maybeRethrowUploadFailure() throws IOException {
            if (this.blockUploadFailure != null) {
                throw this.blockUploadFailure;
            }
        }

        private void uploadBlockAsync(S3ADataBlocks.DataBlock block, Boolean isLast) throws IOException {
            UploadPartRequest request;
            RequestBody requestBody;
            S3ADataBlocks.BlockUploadData uploadData;
            LOG.debug("Queueing upload of {} for upload {}", (Object)block, (Object)this.uploadId);
            Preconditions.checkNotNull(this.uploadId, "Null uploadId");
            this.maybeRethrowUploadFailure();
            ++this.partsSubmitted;
            long size = block.dataSize();
            this.bytesSubmitted += size;
            int currentPartNumber = this.partETagsFutures.size() + 1;
            try {
                uploadData = block.startUpload();
                requestBody = RequestBody.fromContentProvider(uploadData.getContentProvider(), (long)uploadData.getSize(), (String)"application/octet-stream");
                UploadPartRequest.Builder requestBuilder = S3ABlockOutputStream.this.writeOperationHelper.newUploadPartRequestBuilder(S3ABlockOutputStream.this.key, this.uploadId, currentPartNumber, isLast, size);
                request = (UploadPartRequest)((Object)requestBuilder.build());
            }
            catch (SdkException aws) {
                IOException e = S3AUtils.translateException("upload", S3ABlockOutputStream.this.key, aws);
                this.noteUploadFailure(e);
                throw e;
            }
            catch (IOException e) {
                this.noteUploadFailure(e);
                throw e;
            }
            BlockUploadProgress progressCallback = new BlockUploadProgress(block, S3ABlockOutputStream.this.progressListener, S3ABlockOutputStream.this.now());
            S3ABlockOutputStream.this.statistics.blockUploadQueued(block.dataSize());
            Future partETagFuture = S3ABlockOutputStream.this.executorService.submit(() -> {
                try {
                    LOG.debug("Uploading part {} for id '{}'", (Object)currentPartNumber, (Object)this.uploadId);
                    progressCallback.progressChanged(ProgressListenerEvent.TRANSFER_PART_STARTED_EVENT);
                    if (this.uploadAborted.get()) {
                        LOG.debug("Upload of part {} was cancelled", (Object)currentPartNumber);
                        progressCallback.progressChanged(ProgressListenerEvent.TRANSFER_PART_ABORTED_EVENT);
                        CompletedPart completedPart = (CompletedPart)CompletedPart.builder().eTag("").partNumber(currentPartNumber).build();
                        return completedPart;
                    }
                    UploadPartResponse response = S3ABlockOutputStream.this.writeOperationHelper.uploadPart(request, requestBody, S3ABlockOutputStream.this.statistics);
                    LOG.debug("Completed upload of {} to with etag {}", (Object)block, (Object)response.eTag());
                    ++this.partsUploaded;
                    progressCallback.progressChanged(ProgressListenerEvent.TRANSFER_PART_SUCCESS_EVENT);
                    CompletedPart completedPart = (CompletedPart)CompletedPart.builder().eTag(response.eTag()).partNumber(currentPartNumber).checksumCRC32(response.checksumCRC32()).checksumCRC32C(response.checksumCRC32C()).checksumSHA1(response.checksumSHA1()).checksumSHA256(response.checksumSHA256()).build();
                    return completedPart;
                }
                catch (Exception e) {
                    IOException ex = e instanceof IOException ? (IOException)e : new IOException(e);
                    LOG.debug("Failed to upload part {}", (Object)currentPartNumber, (Object)ex);
                    this.noteUploadFailure(ex);
                    progressCallback.progressChanged(ProgressListenerEvent.TRANSFER_PART_FAILED_EVENT);
                    throw ex;
                }
                finally {
                    progressCallback.progressChanged(ProgressListenerEvent.TRANSFER_PART_COMPLETED_EVENT);
                    LOG.debug("closing block");
                    this.completeUpload(currentPartNumber, block, uploadData);
                }
            });
            this.addSubmission(currentPartNumber, block, (ListenableFuture<CompletedPart>)partETagFuture);
        }

        private void addSubmission(int currentPartNumber, S3ADataBlocks.DataBlock block, ListenableFuture<CompletedPart> partETagFuture) {
            this.partETagsFutures.add(partETagFuture);
            this.blocksToClose.put(currentPartNumber, block);
        }

        private void completeUpload(int currentPartNumber, S3ADataBlocks.DataBlock block, S3ADataBlocks.BlockUploadData uploadData) {
            this.blocksToClose.remove(currentPartNumber);
            IOUtils.cleanupWithLogger(LOG, uploadData);
            IOUtils.cleanupWithLogger(LOG, block);
        }

        private List<CompletedPart> waitForAllPartUploads() throws CancellationException, IOException {
            LOG.debug("Waiting for {} uploads to complete", (Object)this.partETagsFutures.size());
            try {
                List<CompletedPart> completedParts = FutureIO.awaitAllFutures(this.partETagsFutures);
                for (CompletedPart part : completedParts) {
                    if (!StringUtils.isEmpty(part.eTag())) continue;
                    throw new CancellationException("Upload of part " + part.partNumber() + " was aborted");
                }
                return completedParts;
            }
            catch (CancellationException e) {
                LOG.warn("Cancelled while waiting for uploads to {} to complete", (Object)S3ABlockOutputStream.this.key, (Object)e);
                throw e;
            }
            catch (IOException | RuntimeException ie) {
                LOG.debug("Failure while waiting for uploads to {} to complete; uploadAborted={}", S3ABlockOutputStream.this.key, this.uploadAborted.get(), ie);
                this.abort();
                throw ie;
            }
        }

        private void cancelAllActiveUploads() {
            LOG.debug("Cancelling {} futures", (Object)this.partETagsFutures.size());
            FutureIO.cancelAllFuturesAndAwaitCompletion(this.partETagsFutures, true, TIME_TO_AWAIT_CANCEL_COMPLETION);
            LOG.debug("Closing blocks");
            this.blocksToClose.forEach((key1, value) -> IOUtils.cleanupWithLogger(LOG, value));
        }

        private void complete(List<CompletedPart> partETags) throws IOException {
            this.maybeRethrowUploadFailure();
            AtomicInteger errorCount = new AtomicInteger(0);
            try {
                IOStatisticsBinding.trackDurationOfInvocation(S3ABlockOutputStream.this.statistics, Statistic.MULTIPART_UPLOAD_COMPLETED.getSymbol(), () -> S3ABlockOutputStream.this.writeOperationHelper.completeMPUwithRetries(S3ABlockOutputStream.this.key, this.uploadId, partETags, this.bytesSubmitted, errorCount, S3ABlockOutputStream.this.builder.putOptions));
            }
            finally {
                S3ABlockOutputStream.this.statistics.exceptionInMultipartComplete(errorCount.get());
            }
        }

        private IOException abort() {
            try {
                if (!this.uploadAborted.getAndSet(true)) {
                    LOG.debug("Aborting upload");
                    S3ABlockOutputStream.this.progressListener.progressChanged(ProgressListenerEvent.TRANSFER_MULTIPART_ABORTED_EVENT, 0L);
                    IOStatisticsBinding.trackDurationOfInvocation(S3ABlockOutputStream.this.statistics, Statistic.OBJECT_MULTIPART_UPLOAD_ABORTED.getSymbol(), () -> {
                        this.cancelAllActiveUploads();
                        S3ABlockOutputStream.this.writeOperationHelper.abortMultipartUpload(S3ABlockOutputStream.this.key, this.uploadId, false, null);
                    });
                }
                return null;
            }
            catch (FileNotFoundException e) {
                return e;
            }
            catch (IOException e) {
                LOG.warn("Unable to abort multipart upload, you may need to purge uploaded parts", e);
                S3ABlockOutputStream.this.statistics.exceptionInMultipartAbort();
                return e;
            }
        }
    }

    private static final class AbortableResultImpl
    implements Abortable.AbortableResult {
        private final boolean alreadyClosed;
        private final IOException anyCleanupException;

        private AbortableResultImpl(boolean alreadyClosed, IOException anyCleanupException) {
            this.alreadyClosed = alreadyClosed;
            this.anyCleanupException = anyCleanupException;
        }

        @Override
        public boolean alreadyClosed() {
            return this.alreadyClosed;
        }

        @Override
        public IOException anyCleanupException() {
            return this.anyCleanupException;
        }

        public String toString() {
            return new StringJoiner(", ", AbortableResultImpl.class.getSimpleName() + "[", "]").add("alreadyClosed=" + this.alreadyClosed).add("anyCleanupException=" + this.anyCleanupException).toString();
        }
    }
}

