/*
 * Decompiled with CFR 0.152.
 */
package com.hazelcast.internal.memory;

import com.hazelcast.internal.memory.NativeMemoryStats;
import com.hazelcast.internal.memory.PooledNativeMemoryStats;
import com.hazelcast.logging.ILogger;
import com.hazelcast.logging.Logger;
import com.hazelcast.memory.Capacity;
import com.hazelcast.memory.NativeOutOfMemoryError;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicLong;

public class MemoryAdjuster {
    public static final ThreadLocal<Boolean> HOT_RESTART_LOADING_IN_PROGRESS = ThreadLocal.withInitial(() -> false);
    static final String PROP_SYSTEM_MEMORY_ENABLED = "hazelcast.internal.system.memory.expansion.enabled";
    static final String DEFAULT_SYSTEM_MEMORY_ENABLED = "true";
    private static final long LOGGING_PERIOD = TimeUnit.MINUTES.toNanos(1L);
    private static final ILogger LOGGER = Logger.getLogger(MemoryAdjuster.class);
    private final boolean systemMemoryEnabled = this.isSystemMemoryEnabled();
    private final NativeMemoryStats stats;
    private final AtomicLong metadataThresholdLastLogTime = new AtomicLong();
    private final AtomicLong systemMemoryExpansionLastLogTime = new AtomicLong();

    public MemoryAdjuster(NativeMemoryStats stats) {
        this.stats = stats;
    }

    private boolean isSystemMemoryEnabled() {
        return Boolean.valueOf(System.getProperty(PROP_SYSTEM_MEMORY_ENABLED, DEFAULT_SYSTEM_MEMORY_ENABLED));
    }

    public void adjustMetadataMemory(long requested) {
        long limit = this.stats.getMaxMetadata();
        long used = this.stats.getUsedMetadata();
        if (used + requested > limit && MemoryAdjuster.shouldLogNow(this.metadataThresholdLastLogTime)) {
            LOGGER.warning(MemoryAdjuster.createRequestedDataMessage(String.format("Allocating more metadata memory than initial threshold of %s!", Capacity.toPrettyString(limit)), requested, this.stats));
        }
        do {
            if (!this.tryIncrementCommitted(requested, AllocationType.METADATA_ALLOCATION)) continue;
            return;
        } while (this.tryIncrementMaxNative(requested, AllocationType.METADATA_ALLOCATION));
        MemoryAdjuster.throwNativeOOME("Metadata allocation request cannot be satisfied! ", requested, this.stats);
    }

    private boolean tryIncrementCommitted(long requested, AllocationType allocationType) {
        long nextCommitted;
        long committed;
        if (requested <= 0L) {
            return false;
        }
        do {
            if ((nextCommitted = (committed = this.stats.getCommittedNative()) + requested) <= MemoryAdjuster.nextCommitThreshold(allocationType, this.stats)) continue;
            return false;
        } while (!this.stats.casCommittedNative(committed, nextCommitted));
        return true;
    }

    private boolean tryIncrementMaxNative(long requested, AllocationType allocationType) {
        long freePhysical;
        long nextMaxNative;
        long maxNative;
        long adjustedSize;
        if (!this.systemMemoryEnabled) {
            return false;
        }
        if (MemoryAdjuster.shouldLogNow(this.systemMemoryExpansionLastLogTime)) {
            LOGGER.warning(MemoryAdjuster.createRequestedDataMessage(String.format("Expanding into system memory to allocate %s.", allocationType.toPrintable()), requested, this.stats));
        }
        long l = adjustedSize = allocationType == AllocationType.METADATA_ALLOCATION ? MemoryAdjuster.fitRequestedToNextPageSize(requested, ((PooledNativeMemoryStats)this.stats).getPageSize()) : requested;
        do {
            if (requested <= (freePhysical = this.stats.getFreePhysical())) continue;
            return false;
        } while (!this.stats.casMaxNative(maxNative = this.stats.getMaxNative(), nextMaxNative = maxNative + (freePhysical >= adjustedSize ? adjustedSize : requested)));
        return true;
    }

    static long fitRequestedToNextPageSize(long requested, long pageSize) {
        long nextPageSize = requested + pageSize;
        return nextPageSize - nextPageSize % pageSize;
    }

    public void adjustDataMemory(long requested) {
        if (MemoryAdjuster.isHotRestartLoadingInProgress()) {
            do {
                if (!this.tryIncrementCommitted(requested, AllocationType.HOT_RESTART_DATA_ALLOCATION)) continue;
                return;
            } while (this.tryIncrementMaxNative(requested, AllocationType.HOT_RESTART_DATA_ALLOCATION));
            MemoryAdjuster.throwNativeOOME("HotRestart data loading cannot be completed! Not enough contiguous memory available!", requested, this.stats);
        }
        if (this.tryIncrementCommitted(requested, AllocationType.REGULAR_DATA_ALLOCATION)) {
            return;
        }
        MemoryAdjuster.throwNativeOOME("Data allocation request cannot be satisfied! Not enough contiguous memory available!", requested, this.stats);
    }

    public void adjustTstoreMemory(long requested) {
        do {
            if (!this.tryIncrementCommitted(requested, AllocationType.TSTORE_DATA_ALLOCATION)) continue;
            return;
        } while (this.tryIncrementMaxNative(requested, AllocationType.TSTORE_DATA_ALLOCATION));
        MemoryAdjuster.throwNativeOOME("Tiered store memory allocation request cannot be satisfied! ", requested, this.stats);
    }

    private static boolean isHotRestartLoadingInProgress() {
        return HOT_RESTART_LOADING_IN_PROGRESS.get();
    }

    private static long nextCommitThreshold(AllocationType allocationType, NativeMemoryStats stats) {
        return allocationType == AllocationType.REGULAR_DATA_ALLOCATION ? stats.getConfiguredMaxNative() : stats.getMaxNative();
    }

    private static void throwNativeOOME(String userGivenMsg, long requested, NativeMemoryStats stats) {
        throw new NativeOutOfMemoryError(MemoryAdjuster.createNoAllocationMessage(userGivenMsg, requested, stats));
    }

    private static String createRequestedDataMessage(String userGivenMsg, long requested, NativeMemoryStats stats) {
        return userGivenMsg + " Requested " + Capacity.toPrettyString(requested) + "!" + MemoryAdjuster.createStatsMessage(stats);
    }

    private static String createNoAllocationMessage(String userGivenMsg, long requested, NativeMemoryStats stats) {
        return userGivenMsg + " Cannot allocate " + Capacity.toPrettyString(requested) + "!" + MemoryAdjuster.createStatsMessage(stats);
    }

    private static String createStatsMessage(NativeMemoryStats stats) {
        String message = " [MaxNative: " + Capacity.toPrettyString(stats.getMaxNative()) + ", CommittedNative: " + Capacity.toPrettyString(stats.getCommittedNative()) + ", UsedNative: " + Capacity.toPrettyString(stats.getUsedNative());
        if (stats instanceof PooledNativeMemoryStats) {
            message = message + ", UsedMetadata: " + Capacity.toPrettyString(stats.getUsedMetadata()) + ", MaxMetadata: " + Capacity.toPrettyString(stats.getMaxMetadata());
        }
        return message + ", FreePhysical: " + Capacity.toPrettyString(stats.getFreePhysical()) + "]";
    }

    private static boolean shouldLogNow(AtomicLong lastLogTime) {
        long lastLogged;
        if (!LOGGER.isWarningEnabled()) {
            return false;
        }
        long now = System.nanoTime();
        if (now - (lastLogged = lastLogTime.get()) >= LOGGING_PERIOD) {
            return lastLogTime.compareAndSet(lastLogged, now);
        }
        return false;
    }

    /*
     * Uses 'sealed' constructs - enablewith --sealed true
     */
    private static enum AllocationType {
        METADATA_ALLOCATION{

            @Override
            String toPrintable() {
                return "metadata";
            }
        }
        ,
        REGULAR_DATA_ALLOCATION{

            @Override
            String toPrintable() {
                return "data";
            }
        }
        ,
        HOT_RESTART_DATA_ALLOCATION{

            @Override
            String toPrintable() {
                return "hot restart data";
            }
        }
        ,
        TSTORE_DATA_ALLOCATION{

            @Override
            String toPrintable() {
                return "tiered store data";
            }
        };


        abstract String toPrintable();
    }
}

