/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.action.bulk;

import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.Phaser;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.function.BiConsumer;
import java.util.function.Predicate;
import java.util.stream.Collectors;
import java.util.stream.StreamSupport;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.elasticsearch.ExceptionsHelper;
import org.elasticsearch.action.ActionListener;
import org.elasticsearch.action.DocWriteRequest;
import org.elasticsearch.action.bulk.BulkItemResponse;
import org.elasticsearch.action.bulk.BulkRequest;
import org.elasticsearch.action.bulk.BulkResponse;
import org.elasticsearch.action.index.IndexRequest;
import org.elasticsearch.common.util.concurrent.EsRejectedExecutionException;
import org.elasticsearch.core.TimeValue;
import org.elasticsearch.rest.RestStatus;

class Retry2 {
    private static final Logger logger = LogManager.getLogger(Retry2.class);
    private final int maxNumberOfRetries;
    private boolean isClosing = false;
    private final Phaser inFlightRequestsPhaser = new Phaser(1);

    Retry2(int maxNumberOfRetries) {
        this.maxNumberOfRetries = maxNumberOfRetries;
    }

    public void consumeRequestWithRetries(BiConsumer<BulkRequest, ActionListener<BulkResponse>> consumer, BulkRequest bulkRequest, ActionListener<BulkResponse> listener) {
        if (this.isClosing) {
            listener.onFailure(new EsRejectedExecutionException("The bulk processor is closing"));
            return;
        }
        ArrayList<BulkItemResponse> responsesAccumulator = new ArrayList<BulkItemResponse>();
        logger.trace("Sending a bulk request with {} bytes in {} items", (Object)bulkRequest.estimatedSizeInBytes(), (Object)bulkRequest.requests.size());
        this.inFlightRequestsPhaser.register();
        consumer.accept(bulkRequest, new RetryHandler(bulkRequest, responsesAccumulator, consumer, listener, this.maxNumberOfRetries));
    }

    private void retry(BulkRequest bulkRequestForRetry, List<BulkItemResponse> responsesAccumulator, BiConsumer<BulkRequest, ActionListener<BulkResponse>> consumer, ActionListener<BulkResponse> listener, int retriesRemaining) {
        if (this.isClosing) {
            listener.onFailure(new EsRejectedExecutionException("The bulk processor is closing"));
            return;
        }
        if (retriesRemaining > 0) {
            this.inFlightRequestsPhaser.register();
            consumer.accept(bulkRequestForRetry, new RetryHandler(bulkRequestForRetry, responsesAccumulator, consumer, listener, retriesRemaining - 1));
        } else {
            listener.onFailure(new EsRejectedExecutionException("Could not retry the bulk request because the backoff policy does not allow any more retries"));
        }
    }

    void awaitClose(long timeout, TimeUnit unit) throws InterruptedException {
        this.isClosing = true;
        this.inFlightRequestsPhaser.arriveAndDeregister();
        try {
            this.inFlightRequestsPhaser.awaitAdvanceInterruptibly(0, timeout, unit);
        }
        catch (TimeoutException e) {
            logger.debug("Timed out waiting for all requests to complete during awaitClose");
        }
    }

    private final class RetryHandler
    implements ActionListener<BulkResponse> {
        private final BulkRequest bulkRequest;
        private final BiConsumer<BulkRequest, ActionListener<BulkResponse>> consumer;
        private final ActionListener<BulkResponse> listener;
        private final List<BulkItemResponse> responsesAccumulator;
        private final long startTimestampNanos;
        private final int retriesRemaining;

        RetryHandler(BulkRequest bulkRequest, List<BulkItemResponse> responsesAccumulator, BiConsumer<BulkRequest, ActionListener<BulkResponse>> consumer, ActionListener<BulkResponse> listener, int retriesRemaining) {
            this.bulkRequest = bulkRequest;
            this.responsesAccumulator = responsesAccumulator;
            this.consumer = consumer;
            this.listener = listener;
            this.startTimestampNanos = System.nanoTime();
            this.retriesRemaining = retriesRemaining;
        }

        @Override
        public void onResponse(BulkResponse bulkItemResponses) {
            if (!bulkItemResponses.hasFailures()) {
                logger.trace("Got a response in {} with {} items, no failures", (Object)bulkItemResponses.getTook(), (Object)bulkItemResponses.getItems().length);
                this.addResponses(bulkItemResponses, r -> true);
                this.listener.onResponse(this.getAccumulatedResponse());
            } else if (this.canRetry(bulkItemResponses)) {
                logger.trace("Got a response in {} with {} items including failures, can retry", (Object)bulkItemResponses.getTook(), (Object)bulkItemResponses.getItems().length);
                this.addResponses(bulkItemResponses, r -> !r.isFailed());
                BulkRequest retryRequest = this.createBulkRequestForRetry(bulkItemResponses);
                Retry2.this.retry(retryRequest, this.responsesAccumulator, this.consumer, this.listener, this.retriesRemaining);
            } else {
                logger.trace("Got a response in {} with {} items including failures, cannot retry", (Object)bulkItemResponses.getTook(), (Object)bulkItemResponses.getItems().length);
                this.addResponses(bulkItemResponses, r -> true);
                this.listener.onResponse(this.getAccumulatedResponse());
            }
            Retry2.this.inFlightRequestsPhaser.arriveAndDeregister();
        }

        @Override
        public void onFailure(Exception e) {
            boolean canRetry;
            boolean bl = canRetry = ExceptionsHelper.status(e) == RestStatus.TOO_MANY_REQUESTS && this.retriesRemaining > 0;
            if (canRetry) {
                Retry2.this.inFlightRequestsPhaser.arriveAndDeregister();
                Retry2.this.retry(this.bulkRequest, this.responsesAccumulator, this.consumer, this.listener, this.retriesRemaining);
            } else {
                this.listener.onFailure(e);
                Retry2.this.inFlightRequestsPhaser.arriveAndDeregister();
            }
        }

        private BulkRequest createBulkRequestForRetry(BulkResponse bulkItemResponses) {
            BulkRequest requestToReissue = new BulkRequest();
            int index = 0;
            for (BulkItemResponse bulkItemResponse : bulkItemResponses.getItems()) {
                if (bulkItemResponse.isFailed()) {
                    DocWriteRequest<?> originalBulkItemRequest = this.bulkRequest.requests().get(index);
                    if (originalBulkItemRequest instanceof IndexRequest) {
                        ((IndexRequest)originalBulkItemRequest).reset();
                    }
                    requestToReissue.add(originalBulkItemRequest);
                }
                ++index;
            }
            return requestToReissue;
        }

        private boolean canRetry(BulkResponse bulkItemResponses) {
            if (this.retriesRemaining == 0) {
                return false;
            }
            for (BulkItemResponse bulkItemResponse : bulkItemResponses) {
                RestStatus status;
                if (!bulkItemResponse.isFailed() || (status = bulkItemResponse.status()) == RestStatus.TOO_MANY_REQUESTS) continue;
                return false;
            }
            return true;
        }

        private void addResponses(BulkResponse response, Predicate<BulkItemResponse> filter) {
            List bulkItemResponses = StreamSupport.stream(response.spliterator(), false).filter(filter).collect(Collectors.toList());
            this.responsesAccumulator.addAll(bulkItemResponses);
        }

        private BulkResponse getAccumulatedResponse() {
            BulkItemResponse[] itemResponses = this.responsesAccumulator.toArray(new BulkItemResponse[0]);
            long stopTimestamp = System.nanoTime();
            long totalLatencyMs = TimeValue.timeValueNanos(stopTimestamp - this.startTimestampNanos).millis();
            logger.trace("Accumulated response includes {} items", (Object)itemResponses.length);
            return new BulkResponse(itemResponses, totalLatencyMs);
        }
    }
}

