/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.fs.azurebfs.services;

import java.io.IOException;
import java.util.Timer;
import java.util.TimerTask;
import java.util.concurrent.TimeUnit;
import org.apache.hadoop.classification.VisibleForTesting;
import org.apache.hadoop.fs.azurebfs.constants.FSOperationType;
import org.apache.hadoop.fs.azurebfs.contracts.exceptions.AzureBlobFileSystemException;
import org.apache.hadoop.fs.azurebfs.services.AbfsClient;
import org.apache.hadoop.fs.azurebfs.services.AbfsRestOperation;
import org.apache.hadoop.fs.azurebfs.utils.TracingContext;
import org.apache.hadoop.io.retry.RetryPolicies;
import org.apache.hadoop.io.retry.RetryPolicy;
import org.apache.hadoop.thirdparty.com.google.common.util.concurrent.FutureCallback;
import org.apache.hadoop.thirdparty.com.google.common.util.concurrent.ListenableScheduledFuture;
import org.apache.hadoop.thirdparty.org.checkerframework.checker.nullness.qual.Nullable;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public final class AbfsLease {
    private static final Logger LOG = LoggerFactory.getLogger(AbfsLease.class);
    static final int DEFAULT_LEASE_ACQUIRE_MAX_RETRIES = 7;
    static final int DEFAULT_LEASE_ACQUIRE_RETRY_INTERVAL = 10;
    private final AbfsClient client;
    private final String path;
    private final TracingContext tracingContext;
    private volatile boolean leaseFreed = false;
    private volatile String leaseID = null;
    private volatile Throwable exception = null;
    private volatile int acquireRetryCount = 0;
    private volatile ListenableScheduledFuture<AbfsRestOperation> future = null;
    private final long leaseRefreshDuration;
    private final int leaseRefreshDurationInSeconds;
    private final Timer timer;
    private LeaseTimerTask leaseTimerTask;
    private final boolean isAsync;

    public AbfsLease(AbfsClient client, String path, boolean isAsync, long leaseRefreshDuration, String eTag, TracingContext tracingContext) throws AzureBlobFileSystemException {
        this(client, path, isAsync, 7, 10, leaseRefreshDuration, eTag, tracingContext);
    }

    @VisibleForTesting
    public AbfsLease(AbfsClient client, String path, boolean isAsync, int acquireMaxRetries, int acquireRetryInterval, long leaseRefreshDuration, String eTag, TracingContext tracingContext) throws AzureBlobFileSystemException {
        this.client = client;
        this.path = path;
        this.tracingContext = tracingContext;
        this.leaseRefreshDuration = leaseRefreshDuration;
        this.leaseRefreshDurationInSeconds = (int)leaseRefreshDuration / 1000;
        this.isAsync = isAsync;
        if (isAsync && client.getNumLeaseThreads() < 1) {
            throw new LeaseException("Lease desired but no lease threads configured, set fs.azure.lease.threads");
        }
        RetryPolicy retryPolicy = RetryPolicies.retryUpToMaximumCountWithFixedSleep(acquireMaxRetries, acquireRetryInterval, TimeUnit.SECONDS);
        this.timer = new Timer(String.format("lease-refresh-timer-%s", path), true);
        this.acquireLease(retryPolicy, 0, acquireRetryInterval, 0L, eTag, new TracingContext(tracingContext));
        while (this.leaseID == null && this.exception == null) {
            try {
                this.future.get();
            }
            catch (Exception e) {
                LOG.debug("Got exception waiting for acquire lease future. Checking if lease ID or exception have been set", e);
            }
        }
        if (this.exception != null) {
            LOG.error("Failed to acquire lease on {}", (Object)path);
            throw new LeaseException(this.exception);
        }
        LOG.debug("Acquired lease {} on {}", (Object)this.leaseID, (Object)path);
    }

    private void acquireLease(final RetryPolicy retryPolicy, final int numRetries, final int retryInterval, long delay, final String eTag, final TracingContext tracingContext) throws LeaseException {
        LOG.debug("Attempting to acquire lease on {}, retry {}", (Object)this.path, (Object)numRetries);
        if (this.future != null && !this.future.isDone()) {
            throw new LeaseException("There is already an existing lease operation");
        }
        FutureCallback<AbfsRestOperation> acquireCallback = new FutureCallback<AbfsRestOperation>(){

            @Override
            public void onSuccess(@Nullable AbfsRestOperation op) {
                AbfsLease.this.leaseID = op.getResult().getResponseHeader("x-ms-lease-id");
                if (AbfsLease.this.leaseRefreshDuration != -1L) {
                    AbfsLease.this.leaseTimerTask = new LeaseTimerTask(AbfsLease.this.client, AbfsLease.this.path, AbfsLease.this.leaseID, tracingContext);
                    AbfsLease.this.timer.scheduleAtFixedRate((TimerTask)AbfsLease.this.leaseTimerTask, AbfsLease.this.leaseRefreshDuration / 2L, AbfsLease.this.leaseRefreshDuration / 2L);
                }
                LOG.debug("Acquired lease {} on {}", (Object)AbfsLease.this.leaseID, (Object)AbfsLease.this.path);
            }

            @Override
            public void onFailure(Throwable throwable) {
                try {
                    if (RetryPolicy.RetryAction.RetryDecision.RETRY == retryPolicy.shouldRetry(null, (int)numRetries, (int)0, (boolean)true).action) {
                        LOG.debug("Failed to acquire lease on {}, retrying: {}", (Object)AbfsLease.this.path, (Object)throwable);
                        AbfsLease.this.acquireRetryCount++;
                        AbfsLease.this.acquireLease(retryPolicy, numRetries + 1, retryInterval, retryInterval, eTag, tracingContext);
                    } else {
                        AbfsLease.this.exception = throwable;
                    }
                }
                catch (Exception e) {
                    AbfsLease.this.exception = throwable;
                }
            }
        };
        if (!this.isAsync) {
            try {
                AbfsRestOperation op = this.client.acquireLease(this.path, this.leaseRefreshDurationInSeconds, eTag, tracingContext);
                acquireCallback.onSuccess(op);
                return;
            }
            catch (AzureBlobFileSystemException ex) {
                acquireCallback.onFailure(ex);
            }
        }
        this.future = this.client.schedule(() -> this.client.acquireLease(this.path, -1, eTag, tracingContext), delay, TimeUnit.SECONDS);
        this.client.addCallback(this.future, acquireCallback);
    }

    public void free() {
        if (this.leaseFreed) {
            return;
        }
        try {
            LOG.debug("Freeing lease: path {}, lease id {}", (Object)this.path, (Object)this.leaseID);
            if (this.future != null && !this.future.isDone()) {
                this.future.cancel(true);
            }
            this.cancelTimer();
            TracingContext tracingContext = new TracingContext(this.tracingContext);
            tracingContext.setOperation(FSOperationType.RELEASE_LEASE);
            this.client.releaseLease(this.path, this.leaseID, tracingContext);
        }
        catch (IOException e) {
            LOG.warn("Exception when trying to release lease {} on {}. Lease will need to be broken: {}", this.leaseID, this.path, e.getMessage());
        }
        finally {
            this.leaseFreed = true;
            LOG.debug("Freed lease {} on {}", (Object)this.leaseID, (Object)this.path);
        }
    }

    public void cancelTimer() {
        if (this.leaseTimerTask != null) {
            this.leaseTimerTask.cancel();
        }
        this.timer.purge();
    }

    public boolean isFreed() {
        return this.leaseFreed;
    }

    public String getLeaseID() {
        return this.leaseID;
    }

    @VisibleForTesting
    public int getAcquireRetryCount() {
        return this.acquireRetryCount;
    }

    @VisibleForTesting
    public TracingContext getTracingContext() {
        return this.tracingContext;
    }

    private static class LeaseTimerTask
    extends TimerTask {
        private final AbfsClient client;
        private final String path;
        private final String leaseID;
        private final TracingContext tracingContext;

        LeaseTimerTask(AbfsClient client, String path, String leaseID, TracingContext tracingContext) {
            this.client = client;
            this.path = path;
            this.leaseID = leaseID;
            this.tracingContext = tracingContext;
        }

        @Override
        public void run() {
            try {
                this.client.renewLease(this.path, this.leaseID, this.tracingContext);
            }
            catch (Exception e) {
                LOG.error("Failed to renew lease on {}", (Object)this.path, (Object)e);
            }
        }
    }

    public static class LeaseException
    extends AzureBlobFileSystemException {
        public LeaseException(Throwable t2) {
            super("Unable to acquire lease: " + t2, t2);
        }

        public LeaseException(String s2) {
            super(s2);
        }
    }
}

