/*
 * Decompiled with CFR 0.152.
 */
package org.wildfly.openssl;

import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import java.util.logging.Level;
import java.util.logging.Logger;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

public final class CipherSuiteConverter {
    private static final Logger LOG = Logger.getLogger(CipherSuiteConverter.class.getName());
    private static final Pattern JAVA_CIPHERSUITE_PATTERN = Pattern.compile("^(?:TLS|SSL)_((?:(?!_WITH_).)+)_WITH_(.*)_(.*)$");
    private static final Pattern OPENSSL_CIPHERSUITE_PATTERN = Pattern.compile("^(?:((?:(?:EXP-)?(?:(?:DHE|EDH|ECDH|ECDHE|SRP)-(?:DSS|RSA|ECDSA)|(?:ADH|AECDH|KRB5|PSK|SRP)))|EXP)-)?(.*)-(.*)$");
    private static final Pattern JAVA_AES_CBC_PATTERN = Pattern.compile("^(AES)_([0-9]+)_CBC$");
    private static final Pattern JAVA_AES_PATTERN = Pattern.compile("^(AES)_([0-9]+)_(.*)$");
    private static final Pattern OPENSSL_AES_CBC_PATTERN = Pattern.compile("^(AES)([0-9]+)$");
    private static final Pattern OPENSSL_AES_PATTERN = Pattern.compile("^(AES)([0-9]+)-(.*)$");
    private static final Pattern OPENSSL_TLSv13_PATTERN = Pattern.compile("^(TLS)_(AES|CHACHA20)_(POLY1305|[0-9]+)_(.*)$");
    private static final ConcurrentMap<String, String> j2o = new ConcurrentHashMap<String, String>();
    private static final ConcurrentMap<String, Map<String, String>> o2j = new ConcurrentHashMap<String, Map<String, String>>();

    static void clearCache() {
        j2o.clear();
        o2j.clear();
    }

    static boolean isJ2OCached(String key, String value) {
        return value.equals(j2o.get(key));
    }

    static boolean isO2JCached(String key, String protocol, String value) {
        Map p2j = (Map)o2j.get(key);
        if (p2j == null) {
            return false;
        }
        return value.equals(p2j.get(protocol));
    }

    public static String toOpenSsl(Iterable<String> javaCipherSuites) {
        StringBuilder buf = new StringBuilder();
        for (String c : javaCipherSuites) {
            if (c == null) break;
            String converted = CipherSuiteConverter.toOpenSsl(c);
            if (converted != null) {
                c = converted;
            }
            buf.append(c);
            buf.append(':');
        }
        if (buf.length() > 0) {
            buf.setLength(buf.length() - 1);
            return buf.toString();
        }
        return "";
    }

    public static String toOpenSsl(String javaCipherSuite) {
        String converted = (String)j2o.get(javaCipherSuite);
        if (converted != null) {
            return converted;
        }
        return CipherSuiteConverter.cacheFromJava(javaCipherSuite);
    }

    private static String cacheFromJava(String javaCipherSuite) {
        String openSslCipherSuite = CipherSuiteConverter.toOpenSslUncached(javaCipherSuite);
        if (openSslCipherSuite == null) {
            return null;
        }
        j2o.putIfAbsent(javaCipherSuite, openSslCipherSuite);
        String javaCipherSuiteSuffix = javaCipherSuite.substring(4);
        HashMap<String, String> p2j = new HashMap<String, String>(4);
        p2j.put("", javaCipherSuiteSuffix);
        p2j.put("SSL", "SSL_" + javaCipherSuiteSuffix);
        p2j.put("TLS", "TLS_" + javaCipherSuiteSuffix);
        o2j.put(openSslCipherSuite, p2j);
        if (LOG.isLoggable(Level.FINE)) {
            LOG.fine("mapping java cipher " + javaCipherSuite + " to " + openSslCipherSuite);
        }
        return openSslCipherSuite;
    }

    static String toOpenSslUncached(String javaCipherSuite) {
        Matcher m4 = JAVA_CIPHERSUITE_PATTERN.matcher(javaCipherSuite);
        if (!m4.matches()) {
            return null;
        }
        String handshakeAlgo = CipherSuiteConverter.toOpenSslHandshakeAlgo(m4.group(1));
        String bulkCipher = CipherSuiteConverter.toOpenSslBulkCipher(m4.group(2));
        String hmacAlgo = CipherSuiteConverter.toOpenSslHmacAlgo(m4.group(3));
        if (handshakeAlgo.length() == 0) {
            return bulkCipher + '-' + hmacAlgo;
        }
        return handshakeAlgo + '-' + bulkCipher + '-' + hmacAlgo;
    }

    private static String toOpenSslHandshakeAlgo(String handshakeAlgo) {
        boolean export = handshakeAlgo.endsWith("_EXPORT");
        if (export) {
            handshakeAlgo = handshakeAlgo.substring(0, handshakeAlgo.length() - 7);
        }
        if ("RSA".equals(handshakeAlgo)) {
            handshakeAlgo = "";
        } else if (handshakeAlgo.endsWith("_anon")) {
            handshakeAlgo = 'A' + handshakeAlgo.substring(0, handshakeAlgo.length() - 5);
        }
        if (export) {
            handshakeAlgo = handshakeAlgo.length() == 0 ? "EXP" : "EXP-" + handshakeAlgo;
        }
        return handshakeAlgo.replace('_', '-');
    }

    private static String toOpenSslBulkCipher(String bulkCipher) {
        if (bulkCipher.startsWith("AES_")) {
            Matcher m4 = JAVA_AES_CBC_PATTERN.matcher(bulkCipher);
            if (m4.matches()) {
                return m4.replaceFirst("$1$2");
            }
            m4 = JAVA_AES_PATTERN.matcher(bulkCipher);
            if (m4.matches()) {
                return m4.replaceFirst("$1$2-$3");
            }
        }
        if ("3DES_EDE_CBC".equals(bulkCipher)) {
            return "DES-CBC3";
        }
        if ("RC4_128".equals(bulkCipher) || "RC4_40".equals(bulkCipher)) {
            return "RC4";
        }
        if ("DES40_CBC".equals(bulkCipher) || "DES_CBC_40".equals(bulkCipher)) {
            return "DES-CBC";
        }
        if ("RC2_CBC_40".equals(bulkCipher)) {
            return "RC2-CBC";
        }
        return bulkCipher.replace('_', '-');
    }

    private static String toOpenSslHmacAlgo(String hmacAlgo) {
        return hmacAlgo;
    }

    public static String toJava(String openSslCipherSuite, String protocol) {
        String javaCipherSuite;
        Map<String, String> p2j = (Map<String, String>)o2j.get(openSslCipherSuite);
        if (p2j == null) {
            p2j = CipherSuiteConverter.cacheFromOpenSsl(openSslCipherSuite);
        }
        if ((javaCipherSuite = p2j.get(protocol)) == null) {
            javaCipherSuite = protocol + '_' + p2j.get("");
        }
        return javaCipherSuite;
    }

    private static Map<String, String> cacheFromOpenSsl(String openSslCipherSuite) {
        String javaCipherSuiteSuffix = CipherSuiteConverter.toJavaUncached(openSslCipherSuite);
        if (javaCipherSuiteSuffix == null) {
            return null;
        }
        String javaCipherSuiteSsl = "SSL_" + javaCipherSuiteSuffix;
        String javaCipherSuiteTls = openSslCipherSuite.startsWith("TLS_") ? javaCipherSuiteSuffix : "TLS_" + javaCipherSuiteSuffix;
        HashMap<String, String> p2j = new HashMap<String, String>(4);
        p2j.put("", javaCipherSuiteSuffix);
        p2j.put("SSL", javaCipherSuiteSsl);
        p2j.put("TLS", javaCipherSuiteTls);
        o2j.putIfAbsent(openSslCipherSuite, p2j);
        j2o.putIfAbsent(javaCipherSuiteTls, openSslCipherSuite);
        j2o.putIfAbsent(javaCipherSuiteSsl, openSslCipherSuite);
        if (LOG.isLoggable(Level.FINE)) {
            LOG.fine("mapping java cipher " + javaCipherSuiteTls + " to " + openSslCipherSuite);
            LOG.fine("mapping java cipher " + javaCipherSuiteSsl + " to " + openSslCipherSuite);
        }
        return p2j;
    }

    static String toJavaUncached(String openSslCipherSuite) {
        String hmacAlgo;
        String bulkCipher;
        boolean export;
        Matcher m4 = OPENSSL_CIPHERSUITE_PATTERN.matcher(openSslCipherSuite);
        if (openSslCipherSuite.startsWith("TLS_")) {
            m4 = OPENSSL_TLSv13_PATTERN.matcher(openSslCipherSuite);
        }
        if (!m4.matches()) {
            return null;
        }
        String handshakeAlgo = m4.group(1);
        if (handshakeAlgo == null) {
            handshakeAlgo = "";
            export = false;
        } else if (handshakeAlgo.startsWith("EXP-")) {
            handshakeAlgo = handshakeAlgo.substring(4);
            export = true;
        } else if ("EXP".equals(handshakeAlgo)) {
            handshakeAlgo = "";
            export = true;
        } else {
            export = false;
        }
        handshakeAlgo = CipherSuiteConverter.toJavaHandshakeAlgo(handshakeAlgo, export);
        if ("TLS".equals(handshakeAlgo)) {
            String groups = m4.group(2) + "_" + m4.group(3);
            bulkCipher = CipherSuiteConverter.toJavaBulkCipher(groups, export);
            hmacAlgo = m4.group(4);
        } else {
            bulkCipher = CipherSuiteConverter.toJavaBulkCipher(m4.group(2), export);
            hmacAlgo = CipherSuiteConverter.toJavaHmacAlgo(m4.group(3));
        }
        if ("TLS".equals(handshakeAlgo)) {
            return handshakeAlgo + "_" + bulkCipher + "_" + hmacAlgo;
        }
        return handshakeAlgo + "_WITH_" + bulkCipher + '_' + hmacAlgo;
    }

    private static String toJavaHandshakeAlgo(String handshakeAlgo, boolean export) {
        if (handshakeAlgo.length() == 0) {
            handshakeAlgo = "RSA";
        } else if ("ADH".equals(handshakeAlgo)) {
            handshakeAlgo = "DH_anon";
        } else if ("AECDH".equals(handshakeAlgo)) {
            handshakeAlgo = "ECDH_anon";
        }
        handshakeAlgo = handshakeAlgo.replace('-', '_');
        if (export) {
            return handshakeAlgo + "_EXPORT";
        }
        return handshakeAlgo;
    }

    private static String toJavaBulkCipher(String bulkCipher, boolean export) {
        if (bulkCipher.startsWith("AES")) {
            Matcher m4 = OPENSSL_AES_CBC_PATTERN.matcher(bulkCipher);
            if (m4.matches()) {
                return m4.replaceFirst("$1_$2_CBC");
            }
            m4 = OPENSSL_AES_PATTERN.matcher(bulkCipher);
            if (m4.matches()) {
                return m4.replaceFirst("$1_$2_$3");
            }
        }
        if ("DES-CBC3".equals(bulkCipher)) {
            return "3DES_EDE_CBC";
        }
        if ("RC4".equals(bulkCipher)) {
            if (export) {
                return "RC4_40";
            }
            return "RC4_128";
        }
        if ("DES-CBC".equals(bulkCipher)) {
            if (export) {
                return "DES_CBC_40";
            }
            return "DES_CBC";
        }
        if ("RC2-CBC".equals(bulkCipher)) {
            if (export) {
                return "RC2_CBC_40";
            }
            return "RC2_CBC";
        }
        return bulkCipher.replace('-', '_');
    }

    private static String toJavaHmacAlgo(String hmacAlgo) {
        return hmacAlgo;
    }

    private CipherSuiteConverter() {
    }
}

