/*
 * Decompiled with CFR 0.152.
 */
package com.microsoft.azure.datalake.store;

import com.microsoft.azure.datalake.store.ADLFileInputStream;
import com.microsoft.azure.datalake.store.ReadBuffer;
import com.microsoft.azure.datalake.store.ReadBufferStatus;
import com.microsoft.azure.datalake.store.ReadBufferWorker;
import java.util.Collection;
import java.util.LinkedList;
import java.util.Queue;
import java.util.Stack;
import java.util.concurrent.CountDownLatch;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

class ReadBufferManager {
    private static final Logger log = LoggerFactory.getLogger("com.microsoft.azure.datalake.store.ReadBufferManager");
    private static final int numBuffers = 16;
    private static final int blocksize = 0x400000;
    private static final int numThreads = 8;
    private static final int thresholdAgeMilliseconds = 3000;
    private Thread[] threads = new Thread[8];
    private byte[][] buffers;
    private Stack<Integer> freeList = new Stack();
    private Queue<ReadBuffer> readAheadQueue = new LinkedList<ReadBuffer>();
    private LinkedList<ReadBuffer> inProgressList = new LinkedList();
    private LinkedList<ReadBuffer> completedReadList = new LinkedList();
    private static final ReadBufferManager bufferManager = new ReadBufferManager();

    static ReadBufferManager getBufferManager() {
        return bufferManager;
    }

    private void init() {
        int i;
        this.buffers = new byte[16][];
        for (i = 0; i < 16; ++i) {
            this.buffers[i] = new byte[0x400000];
            this.freeList.add(i);
        }
        for (i = 0; i < 8; ++i) {
            Thread t2 = new Thread(new ReadBufferWorker(i));
            t2.setDaemon(true);
            this.threads[i] = t2;
            t2.setName("ADLS-prefetch-" + i);
            t2.start();
        }
        ReadBufferWorker.unleashWorkers.countDown();
    }

    private ReadBufferManager() {
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    void queueReadAhead(ADLFileInputStream file, long requestedOffset, int requestedLength) {
        ReadBuffer buffer;
        if (log.isTraceEnabled()) {
            log.trace("Start Queueing readAhead for " + file.getFilename() + " offset " + requestedOffset + " length " + requestedLength);
        }
        ReadBufferManager readBufferManager = this;
        synchronized (readBufferManager) {
            if (this.isAlreadyQueued(file, requestedOffset)) {
                return;
            }
            if (this.freeList.size() == 0 && !this.tryEvict()) {
                return;
            }
            buffer = new ReadBuffer();
            buffer.file = file;
            buffer.offset = requestedOffset;
            buffer.length = 0;
            buffer.requestedLength = requestedLength;
            buffer.status = ReadBufferStatus.NOT_AVAILABLE;
            buffer.latch = new CountDownLatch(1);
            Integer bufferIndex = this.freeList.pop();
            buffer.buffer = this.buffers[bufferIndex];
            buffer.bufferindex = bufferIndex;
            this.readAheadQueue.add(buffer);
            this.notifyAll();
        }
        if (log.isTraceEnabled()) {
            log.trace("Done q-ing readAhead for file " + file.getFilename() + " offset " + requestedOffset + " buffer idx " + buffer.bufferindex);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    int getBlock(ADLFileInputStream file, long position, int length, byte[] buffer) {
        ReadBuffer readBuf;
        if (log.isTraceEnabled()) {
            log.trace("getBlock for file " + file.getFilename() + " position " + position + " thread " + Thread.currentThread().getName());
        }
        ReadBufferManager readBufferManager = this;
        synchronized (readBufferManager) {
            this.clearFromReadAheadQueue(file, position);
            readBuf = this.getFromList(this.inProgressList, file, position);
        }
        if (readBuf != null) {
            try {
                if (log.isTraceEnabled()) {
                    log.trace("got a relevant read buffer for file " + file.getFilename() + " offset " + readBuf.offset + " buffer idx " + readBuf.bufferindex);
                }
                readBuf.latch.await();
            }
            catch (InterruptedException ex) {
                Thread.currentThread().interrupt();
            }
            if (log.isTraceEnabled()) {
                log.trace("latch done for file " + file.getFilename() + " buffer idx " + readBuf.bufferindex + " length " + readBuf.length);
            }
        }
        int bytesRead = 0;
        readBufferManager = this;
        synchronized (readBufferManager) {
            bytesRead = this.getBlockFromCompletedQueue(file, position, length, buffer);
        }
        if (bytesRead > 0) {
            if (log.isTraceEnabled()) {
                log.trace("Done read from Cache for " + file.getFilename() + " position " + position + " length " + bytesRead);
            }
            return bytesRead;
        }
        return 0;
    }

    private synchronized boolean tryEvict() {
        ReadBuffer nodeToEvict = null;
        if (this.completedReadList.size() <= 0) {
            return false;
        }
        for (ReadBuffer buf : this.completedReadList) {
            if (!buf.firstByteConsumed || !buf.lastByteConsumed) continue;
            nodeToEvict = buf;
            break;
        }
        if (nodeToEvict != null) {
            return this.evict(nodeToEvict);
        }
        for (ReadBuffer buf : this.completedReadList) {
            if (!buf.anyByteConsumed) continue;
            nodeToEvict = buf;
            break;
        }
        if (nodeToEvict != null) {
            return this.evict(nodeToEvict);
        }
        long earliestBirthday = Long.MAX_VALUE;
        for (ReadBuffer buf : this.completedReadList) {
            if (buf.birthday >= earliestBirthday) continue;
            nodeToEvict = buf;
            earliestBirthday = buf.birthday;
        }
        if (this.currentTimeMillis() - earliestBirthday > 3000L && nodeToEvict != null) {
            return this.evict(nodeToEvict);
        }
        return false;
    }

    private boolean evict(ReadBuffer buf) {
        this.freeList.push(buf.bufferindex);
        this.completedReadList.remove(buf);
        if (log.isTraceEnabled()) {
            log.trace("Evicting buffer idx " + buf.bufferindex + "; was used for file " + buf.file.getFilename() + " offset " + buf.offset + " length " + buf.length);
        }
        return true;
    }

    private boolean isAlreadyQueued(ADLFileInputStream file, long requestedOffset) {
        return this.isInList(this.readAheadQueue, file, requestedOffset) || this.isInList(this.inProgressList, file, requestedOffset) || this.isInList(this.completedReadList, file, requestedOffset);
    }

    private boolean isInList(Collection<ReadBuffer> list, ADLFileInputStream file, long requestedOffset) {
        return this.getFromList(list, file, requestedOffset) != null;
    }

    private ReadBuffer getFromList(Collection<ReadBuffer> list, ADLFileInputStream file, long requestedOffset) {
        for (ReadBuffer buffer : list) {
            if (buffer.file != file) continue;
            if (buffer.status == ReadBufferStatus.AVAILABLE && requestedOffset >= buffer.offset && requestedOffset < buffer.offset + (long)buffer.length) {
                return buffer;
            }
            if (requestedOffset < buffer.offset || requestedOffset >= buffer.offset + (long)buffer.requestedLength) continue;
            return buffer;
        }
        return null;
    }

    private void clearFromReadAheadQueue(ADLFileInputStream file, long requestedOffset) {
        ReadBuffer buffer = this.getFromList(this.readAheadQueue, file, requestedOffset);
        if (buffer != null) {
            this.readAheadQueue.remove(buffer);
            this.notifyAll();
            this.freeList.push(buffer.bufferindex);
        }
    }

    private int getBlockFromCompletedQueue(ADLFileInputStream file, long position, int length, byte[] buffer) {
        ReadBuffer buf = this.getFromList(this.completedReadList, file, position);
        if (buf == null || position >= buf.offset + (long)buf.length) {
            return 0;
        }
        int cursor = (int)(position - buf.offset);
        int availableLengthInBuffer = buf.length - cursor;
        int lengthToCopy = Math.min(length, availableLengthInBuffer);
        System.arraycopy(buf.buffer, cursor, buffer, 0, lengthToCopy);
        if (cursor == 0) {
            buf.firstByteConsumed = true;
        }
        if (cursor + lengthToCopy == buf.length) {
            buf.lastByteConsumed = true;
        }
        buf.anyByteConsumed = true;
        return lengthToCopy;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    ReadBuffer getNextBlockToRead() throws InterruptedException {
        ReadBuffer buffer = null;
        ReadBufferManager readBufferManager = this;
        synchronized (readBufferManager) {
            while (this.readAheadQueue.size() == 0) {
                this.wait();
            }
            buffer = this.readAheadQueue.remove();
            this.notifyAll();
            if (buffer == null) {
                return null;
            }
            buffer.status = ReadBufferStatus.READING_IN_PROGRESS;
            this.inProgressList.add(buffer);
        }
        if (log.isTraceEnabled()) {
            log.trace("ReadBufferWorker picked file " + buffer.file.getFilename() + " for offset " + buffer.offset);
        }
        return buffer;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    void doneReading(ReadBuffer buffer, ReadBufferStatus result, int bytesActuallyRead) {
        if (log.isTraceEnabled()) {
            log.trace("ReadBufferWorker completed file " + buffer.file.getFilename() + " for offset " + buffer.offset + " bytes " + bytesActuallyRead);
        }
        ReadBufferManager readBufferManager = this;
        synchronized (readBufferManager) {
            this.inProgressList.remove(buffer);
            if (result == ReadBufferStatus.AVAILABLE && bytesActuallyRead > 0) {
                buffer.status = ReadBufferStatus.AVAILABLE;
                buffer.birthday = this.currentTimeMillis();
                buffer.length = bytesActuallyRead;
                this.completedReadList.add(buffer);
            } else {
                this.freeList.push(buffer.bufferindex);
            }
        }
        buffer.latch.countDown();
    }

    private long currentTimeMillis() {
        return System.nanoTime() / 1000L / 1000L;
    }

    static {
        bufferManager.init();
    }
}

