/*
 * Decompiled with CFR 0.152.
 */
package com.hazelcast.internal.memory.impl;

import com.hazelcast.config.NativeMemoryConfig;
import com.hazelcast.internal.memory.impl.LibMalloc;
import com.hazelcast.internal.memory.impl.MemkindHeap;
import com.hazelcast.internal.memory.impl.NUMA;
import com.hazelcast.internal.memory.impl.PersistentMemoryDirectory;
import com.hazelcast.internal.util.JVMUtil;
import com.hazelcast.internal.util.OsHelper;
import com.hazelcast.internal.util.Preconditions;
import com.hazelcast.logging.ILogger;
import com.hazelcast.logging.Logger;
import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.atomic.AtomicLong;
import java.util.stream.Collectors;

final class MemkindPmemMalloc
implements LibMalloc {
    private static final ILogger LOGGER = Logger.getLogger(MemkindPmemMalloc.class);
    private final ThreadLocal<AllocationStrategy> cachedAllocationStrategy = new ThreadLocal();
    private final List<PersistentMemoryDirectory> directories;
    private final ArrayList<MemkindHeap> heaps;
    private final AtomicLong heapSeq = new AtomicLong();
    private final AllocationStrategy roundRobinAllocationStrategy;
    private final boolean numaEnabled;

    private MemkindPmemMalloc(ArrayList<MemkindHeap> heaps, List<PersistentMemoryDirectory> directories) {
        Preconditions.checkTrue(!heaps.isEmpty(), "Heaps must not be empty");
        this.heaps = heaps;
        this.directories = directories;
        this.roundRobinAllocationStrategy = () -> (MemkindHeap)heaps.get(this.nextHeapIndex());
        boolean numaConfigured = true;
        int maxNumaNodeConfigured = -1;
        for (MemkindHeap heap : heaps) {
            int numaNode = heap.getNumaNode();
            numaConfigured &= numaNode >= 0;
            if (numaNode <= maxNumaNodeConfigured) continue;
            maxNumaNodeConfigured = numaNode;
        }
        this.numaEnabled = numaConfigured && this.isNumaEnabled(maxNumaNodeConfigured);
    }

    private int nextHeapIndex() {
        return (int)(this.heapSeq.getAndIncrement() % (long)this.heaps.size());
    }

    private int currentHeapIndex() {
        return (int)(this.heapSeq.get() % (long)this.heaps.size());
    }

    private boolean isNumaEnabled(int maxNumaNodeConfigured) {
        boolean numaEnabled;
        int maxNumaNode;
        boolean numaAvailable = NUMA.available();
        int n = maxNumaNode = numaAvailable ? NUMA.maxNumaNode() : 0;
        if (!numaAvailable) {
            LOGGER.warning("NUMA nodes are configured for the persistent memory directories but the NUMA functions are not available. NUMA awareness is disabled.");
            numaEnabled = false;
        } else if (maxNumaNodeConfigured > maxNumaNode) {
            LOGGER.warning(String.format("Invalid NUMA configuration: greatest available NUMA node in the system is '%d', greatest NUMA node specified in the persistent memory configuration: '%d'. NUMA awareness is disabled.", maxNumaNode, maxNumaNodeConfigured));
            numaEnabled = false;
        } else {
            LOGGER.info("Allocation strategy preferring NUMA-local allocations is enabled for the persistent memory backed allocations");
            numaEnabled = true;
        }
        return numaEnabled;
    }

    static MemkindPmemMalloc create(NativeMemoryConfig config, long size) {
        MemkindPmemMalloc.checkPlatform();
        Preconditions.checkTrue(!config.getPersistentMemoryConfig().getDirectoryConfigs().isEmpty(), "At least one persistent memory directory needs to be configured to use the persistent memory allocator.");
        List<PersistentMemoryDirectory> pmemDirectories = config.getPersistentMemoryConfig().getDirectoryConfigs().stream().map(PersistentMemoryDirectory::new).collect(Collectors.toList());
        MemkindPmemMalloc.logPmemDirectories(pmemDirectories);
        ArrayList<MemkindHeap> heaps = new ArrayList<MemkindHeap>(pmemDirectories.size());
        long singleHeapSize = size / (long)pmemDirectories.size();
        try {
            for (PersistentMemoryDirectory directory : pmemDirectories) {
                String pmemFilePath = directory.getPersistentMemoryFile().getAbsolutePath();
                heaps.add(MemkindHeap.createPmemHeap(pmemFilePath, singleHeapSize, directory.getNumaNodeId()));
            }
            return new MemkindPmemMalloc(heaps, pmemDirectories);
        }
        catch (Exception ex) {
            MemkindPmemMalloc.cleanUp(heaps, pmemDirectories);
            throw ex;
        }
    }

    private static void logPmemDirectories(List<PersistentMemoryDirectory> pmemDirectories) {
        StringBuilder sb = new StringBuilder("Using Memkind PMEM memory allocator with paths:\n");
        pmemDirectories.forEach(pmemDir -> sb.append("\t- ").append(pmemDir.getPersistentMemoryFile().getAbsolutePath()).append(", configured NUMA node: ").append(pmemDir.getNumaNodeId()).append("\n"));
        LOGGER.info(sb.toString());
    }

    static void checkPlatform() {
        if (!OsHelper.isLinux()) {
            throw new UnsupportedOperationException("Persistent memory is not supported on this platform: " + OsHelper.OS + ". Only Linux platform is supported.");
        }
        if (JVMUtil.is32bitJVM()) {
            throw new UnsupportedOperationException("Persistent memory is not supported on 32 bit JVM");
        }
    }

    public String toString() {
        return "MemkindPmemMalloc";
    }

    @Override
    public long malloc(long size) {
        long address2;
        MemkindHeap heap = this.takeHeap();
        try {
            address2 = heap.allocate(size);
            if (address2 != 0L) {
                return address2;
            }
        }
        catch (OutOfMemoryError address2) {
            // empty catch block
        }
        address2 = 0L;
        int startHeapIndex = this.currentHeapIndex();
        for (int i = 0; i < this.heaps.size() && address2 == 0L; ++i) {
            int heapIndex = (startHeapIndex + i) % this.heaps.size();
            MemkindHeap overflowHeap = this.heaps.get(heapIndex);
            if (overflowHeap == heap) continue;
            try {
                address2 = overflowHeap.allocate(size);
                continue;
            }
            catch (OutOfMemoryError outOfMemoryError) {
                // empty catch block
            }
        }
        return address2;
    }

    @Override
    public long realloc(long address, long size) {
        try {
            return this.aHeap().realloc(address, size);
        }
        catch (OutOfMemoryError e) {
            return 0L;
        }
    }

    @Override
    public void free(long address) {
        this.aHeap().free(address);
    }

    @Override
    public void dispose() {
        MemkindPmemMalloc.cleanUp(this.heaps, this.directories);
    }

    private static void cleanUp(List<MemkindHeap> heaps, List<PersistentMemoryDirectory> directories) {
        boolean closedHeaps = MemkindPmemMalloc.closeHeaps(heaps);
        boolean disposeDirectories = MemkindPmemMalloc.disposeDirectories(directories);
        if (!closedHeaps || !disposeDirectories) {
            LOGGER.warning("Could not properly clean up the used file system resources.");
        }
    }

    private static boolean disposeDirectories(List<PersistentMemoryDirectory> directories) {
        boolean disposedDirectories = true;
        for (PersistentMemoryDirectory directory : directories) {
            try {
                directory.dispose();
            }
            catch (Exception ex) {
                LOGGER.severe("Could not dispose PMEM directory " + directory, ex);
                disposedDirectories = false;
            }
        }
        return disposedDirectories;
    }

    private static boolean closeHeaps(List<MemkindHeap> heaps) {
        boolean closedHeaps = true;
        for (MemkindHeap heap : heaps) {
            try {
                heap.close();
            }
            catch (Exception ex) {
                LOGGER.severe("Could not close heap " + heap, ex);
                closedHeaps = false;
            }
        }
        return closedHeaps;
    }

    private MemkindHeap takeHeap() {
        int boundedNumaNode;
        AllocationStrategy allocationStrategy = this.cachedAllocationStrategy.get();
        if (allocationStrategy != null) {
            return allocationStrategy.takeHeap();
        }
        if (this.numaEnabled && (boundedNumaNode = NUMA.currentThreadBoundedNumaNode()) != -1) {
            for (MemkindHeap heap : this.heaps) {
                int heapNumaNode = heap.getNumaNode();
                if (boundedNumaNode != heapNumaNode) continue;
                if (LOGGER.isFineEnabled()) {
                    String currentThread = Thread.currentThread().getName();
                    LOGGER.fine(String.format("Cached heap '%s' assigned to NUMA node '%d' for thread '%s'", heap.getName(), heap.getNumaNode(), currentThread));
                }
                this.cachedAllocationStrategy.set(() -> heap);
                return heap;
            }
        }
        this.cachedAllocationStrategy.set(this.roundRobinAllocationStrategy);
        if (LOGGER.isFineEnabled()) {
            String threadName = Thread.currentThread().getName();
            String message = String.format("Using round-robin heap allocation strategy for thread %s", threadName);
            LOGGER.fine(message);
        }
        return this.roundRobinAllocationStrategy.takeHeap();
    }

    public MemkindHeap aHeap() {
        return this.heaps.get(0);
    }

    @FunctionalInterface
    private static interface AllocationStrategy {
        public MemkindHeap takeHeap();
    }
}

