/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.tasks;

import java.io.IOException;
import java.util.Collection;
import java.util.List;
import java.util.Objects;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.apache.logging.log4j.message.Message;
import org.apache.logging.log4j.message.ParameterizedMessage;
import org.elasticsearch.ElasticsearchSecurityException;
import org.elasticsearch.ExceptionsHelper;
import org.elasticsearch.Version;
import org.elasticsearch.action.ActionListener;
import org.elasticsearch.action.ResultDeduplicator;
import org.elasticsearch.action.StepListener;
import org.elasticsearch.action.support.ChannelActionListener;
import org.elasticsearch.action.support.GroupedActionListener;
import org.elasticsearch.common.io.stream.StreamInput;
import org.elasticsearch.common.io.stream.StreamOutput;
import org.elasticsearch.tasks.CancellableTask;
import org.elasticsearch.tasks.Task;
import org.elasticsearch.tasks.TaskId;
import org.elasticsearch.tasks.TaskManager;
import org.elasticsearch.transport.EmptyTransportResponseHandler;
import org.elasticsearch.transport.NodeDisconnectedException;
import org.elasticsearch.transport.NodeNotConnectedException;
import org.elasticsearch.transport.Transport;
import org.elasticsearch.transport.TransportChannel;
import org.elasticsearch.transport.TransportException;
import org.elasticsearch.transport.TransportRequest;
import org.elasticsearch.transport.TransportRequestHandler;
import org.elasticsearch.transport.TransportRequestOptions;
import org.elasticsearch.transport.TransportResponse;
import org.elasticsearch.transport.TransportService;

public class TaskCancellationService {
    public static final String BAN_PARENT_ACTION_NAME = "internal:admin/tasks/ban";
    private static final Logger logger = LogManager.getLogger(TaskCancellationService.class);
    private final TransportService transportService;
    private final TaskManager taskManager;
    private final ResultDeduplicator<CancelRequest, Void> deduplicator = new ResultDeduplicator();

    public TaskCancellationService(TransportService transportService) {
        this.transportService = transportService;
        this.taskManager = transportService.getTaskManager();
        transportService.registerRequestHandler(BAN_PARENT_ACTION_NAME, "same", x$0 -> new BanParentTaskRequest(x$0), new BanParentRequestHandler());
    }

    private String localNodeId() {
        return this.transportService.getLocalNode().getId();
    }

    void cancelTaskAndDescendants(CancellableTask task, String reason, boolean waitForCompletion, ActionListener<Void> finalListener) {
        this.deduplicator.executeOnce(new CancelRequest(task, waitForCompletion), finalListener, (r, listener) -> this.doCancelTaskAndDescendants(task, reason, waitForCompletion, (ActionListener<Void>)listener));
    }

    void doCancelTaskAndDescendants(CancellableTask task, String reason, boolean waitForCompletion, ActionListener<Void> listener) {
        TaskId taskId = task.taskInfo(this.localNodeId(), false).getTaskId();
        if (task.shouldCancelChildrenOnCancellation()) {
            logger.trace("cancelling task [{}] and its descendants", (Object)taskId);
            StepListener<Void> completedListener = new StepListener<Void>();
            GroupedActionListener groupedListener = new GroupedActionListener(completedListener.map(r -> null), 3);
            Collection<Transport.Connection> childConnections = this.taskManager.startBanOnChildTasks(task.getId(), reason, () -> {
                logger.trace("child tasks of parent [{}] are completed", (Object)taskId);
                groupedListener.onResponse(null);
            });
            this.taskManager.cancel(task, reason, () -> {
                logger.trace("task [{}] is cancelled", (Object)taskId);
                groupedListener.onResponse(null);
            });
            StepListener<Void> setBanListener = new StepListener<Void>();
            this.setBanOnChildConnections(reason, waitForCompletion, task, childConnections, setBanListener);
            setBanListener.addListener(groupedListener);
            Runnable removeBansRunnable = this.transportService.getThreadPool().getThreadContext().preserveContext(() -> this.removeBanOnChildConnections(task, childConnections));
            completedListener.whenComplete(r -> removeBansRunnable.run(), e -> removeBansRunnable.run());
            if (waitForCompletion) {
                completedListener.addListener(listener);
            } else {
                setBanListener.addListener(listener);
            }
        } else {
            logger.trace("task [{}] doesn't have any children that should be cancelled", (Object)taskId);
            if (waitForCompletion) {
                this.taskManager.cancel(task, reason, () -> listener.onResponse(null));
            } else {
                this.taskManager.cancel(task, reason, () -> {});
                listener.onResponse(null);
            }
        }
    }

    private void setBanOnChildConnections(String reason, boolean waitForCompletion, CancellableTask task, Collection<Transport.Connection> childConnections, ActionListener<Void> listener) {
        if (childConnections.isEmpty()) {
            listener.onResponse(null);
            return;
        }
        final TaskId taskId = new TaskId(this.localNodeId(), task.getId());
        logger.trace("cancelling child tasks of [{}] on child connections {}", (Object)taskId, childConnections);
        final GroupedActionListener groupedListener = new GroupedActionListener(listener.map(r -> null), childConnections.size());
        BanParentTaskRequest banRequest = BanParentTaskRequest.createSetBanParentTaskRequest(taskId, reason, waitForCompletion);
        for (final Transport.Connection connection : childConnections) {
            assert (TransportService.unwrapConnection(connection) == connection) : "Child connection must be unwrapped";
            this.transportService.sendRequest(connection, BAN_PARENT_ACTION_NAME, (TransportRequest)banRequest, TransportRequestOptions.EMPTY, new EmptyTransportResponseHandler("same"){

                @Override
                public void handleResponse(TransportResponse.Empty response) {
                    logger.trace("sent ban for tasks with the parent [{}] for connection [{}]", (Object)taskId, (Object)connection);
                    groupedListener.onResponse(null);
                }

                @Override
                public void handleException(TransportException exp) {
                    Throwable cause = ExceptionsHelper.unwrapCause(exp);
                    assert (!(cause instanceof ElasticsearchSecurityException));
                    if (TaskCancellationService.isUnimportantBanFailure(cause)) {
                        logger.debug((Message)new ParameterizedMessage("cannot send ban for tasks with the parent [{}] on connection [{}]", (Object)taskId, (Object)connection), (Throwable)exp);
                    } else if (logger.isDebugEnabled()) {
                        logger.warn((Message)new ParameterizedMessage("cannot send ban for tasks with the parent [{}] on connection [{}]", (Object)taskId, (Object)connection), (Throwable)exp);
                    } else {
                        logger.warn("cannot send ban for tasks with the parent [{}] on connection [{}]: {}", (Object)taskId, (Object)connection, (Object)exp.getMessage());
                    }
                    groupedListener.onFailure(exp);
                }
            });
        }
    }

    private void removeBanOnChildConnections(CancellableTask task, Collection<Transport.Connection> childConnections) {
        final BanParentTaskRequest request = BanParentTaskRequest.createRemoveBanParentTaskRequest(new TaskId(this.localNodeId(), task.getId()));
        for (final Transport.Connection connection : childConnections) {
            assert (TransportService.unwrapConnection(connection) == connection) : "Child connection must be unwrapped";
            logger.trace("Sending remove ban for tasks with the parent [{}] for connection [{}]", (Object)request.parentTaskId, (Object)connection);
            this.transportService.sendRequest(connection, BAN_PARENT_ACTION_NAME, (TransportRequest)request, TransportRequestOptions.EMPTY, new EmptyTransportResponseHandler("same"){

                @Override
                public void handleException(TransportException exp) {
                    Throwable cause = ExceptionsHelper.unwrapCause(exp);
                    assert (!(cause instanceof ElasticsearchSecurityException));
                    if (TaskCancellationService.isUnimportantBanFailure(cause)) {
                        logger.debug((Message)new ParameterizedMessage("failed to remove ban for tasks with the parent [{}] on connection [{}]", (Object)request.parentTaskId, (Object)connection), (Throwable)exp);
                    } else if (logger.isDebugEnabled()) {
                        logger.warn((Message)new ParameterizedMessage("failed to remove ban for tasks with the parent [{}] on connection [{}]", (Object)request.parentTaskId, (Object)connection), (Throwable)exp);
                    } else {
                        logger.warn("failed to remove ban for tasks with the parent [{}] on connection [{}]: {}", (Object)request.parentTaskId, (Object)connection, (Object)exp.getMessage());
                    }
                }
            });
        }
    }

    private static boolean isUnimportantBanFailure(Throwable cause) {
        return cause instanceof NodeDisconnectedException || cause instanceof NodeNotConnectedException;
    }

    private class BanParentRequestHandler
    implements TransportRequestHandler<BanParentTaskRequest> {
        private BanParentRequestHandler() {
        }

        @Override
        public void messageReceived(BanParentTaskRequest request, TransportChannel channel, Task task) throws Exception {
            if (request.ban) {
                logger.debug("Received ban for the parent [{}] on the node [{}], reason: [{}]", (Object)request.parentTaskId, (Object)TaskCancellationService.this.localNodeId(), (Object)request.reason);
                List<CancellableTask> childTasks = TaskCancellationService.this.taskManager.setBan(request.parentTaskId, request.reason, channel);
                GroupedActionListener<Void> listener = new GroupedActionListener<Void>(new ChannelActionListener<TransportResponse, BanParentTaskRequest>(channel, TaskCancellationService.BAN_PARENT_ACTION_NAME, request).map(r -> TransportResponse.Empty.INSTANCE), childTasks.size() + 1);
                for (CancellableTask childTask : childTasks) {
                    TaskCancellationService.this.cancelTaskAndDescendants(childTask, request.reason, request.waitForCompletion, listener);
                }
                listener.onResponse(null);
            } else {
                logger.debug("Removing ban for the parent [{}] on the node [{}]", (Object)request.parentTaskId, (Object)TaskCancellationService.this.localNodeId());
                TaskCancellationService.this.taskManager.removeBan(request.parentTaskId);
                channel.sendResponse(TransportResponse.Empty.INSTANCE);
            }
        }
    }

    private static class CancelRequest {
        final CancellableTask task;
        final boolean waitForCompletion;

        CancelRequest(CancellableTask task, boolean waitForCompletion) {
            this.task = task;
            this.waitForCompletion = waitForCompletion;
        }

        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || this.getClass() != o.getClass()) {
                return false;
            }
            CancelRequest that = (CancelRequest)o;
            return this.waitForCompletion == that.waitForCompletion && Objects.equals(this.task, that.task);
        }

        public int hashCode() {
            return Objects.hash(this.task, this.waitForCompletion);
        }
    }

    private static class BanParentTaskRequest
    extends TransportRequest {
        private final TaskId parentTaskId;
        private final boolean ban;
        private final boolean waitForCompletion;
        private final String reason;

        static BanParentTaskRequest createSetBanParentTaskRequest(TaskId parentTaskId, String reason, boolean waitForCompletion) {
            return new BanParentTaskRequest(parentTaskId, reason, waitForCompletion);
        }

        static BanParentTaskRequest createRemoveBanParentTaskRequest(TaskId parentTaskId) {
            return new BanParentTaskRequest(parentTaskId);
        }

        private BanParentTaskRequest(TaskId parentTaskId, String reason, boolean waitForCompletion) {
            this.parentTaskId = parentTaskId;
            this.ban = true;
            this.reason = reason;
            this.waitForCompletion = waitForCompletion;
        }

        private BanParentTaskRequest(TaskId parentTaskId) {
            this.parentTaskId = parentTaskId;
            this.ban = false;
            this.reason = null;
            this.waitForCompletion = false;
        }

        private BanParentTaskRequest(StreamInput in) throws IOException {
            super(in);
            this.parentTaskId = TaskId.readFromStream(in);
            this.ban = in.readBoolean();
            this.reason = this.ban ? in.readString() : null;
            this.waitForCompletion = in.getVersion().onOrAfter(Version.V_7_8_0) ? in.readBoolean() : false;
        }

        @Override
        public void writeTo(StreamOutput out) throws IOException {
            super.writeTo(out);
            this.parentTaskId.writeTo(out);
            out.writeBoolean(this.ban);
            if (this.ban) {
                out.writeString(this.reason);
            }
            if (out.getVersion().onOrAfter(Version.V_7_8_0)) {
                out.writeBoolean(this.waitForCompletion);
            }
        }
    }
}

